package com.instabug.library.internal.filestore

import com.instabug.library.util.extenstions.createNewFileDefensive
import com.instabug.library.util.extenstions.deleteDefensive
import com.instabug.library.util.extenstions.deleteRecursivelyDefensive
import com.instabug.library.util.extenstions.getOrLogAndReport
import com.instabug.library.util.extenstions.logDWithThreadName
import com.instabug.library.util.extenstions.logVWithThreadName
import com.instabug.library.util.extenstions.mkdirDefensive
import com.instabug.library.util.extenstions.mkdirsDefensive
import com.instabug.library.util.extenstions.prefixedWithT
import com.instabug.library.util.extenstions.readJSONLines
import com.instabug.library.util.extenstions.runOrLogAndReport
import com.instabug.library.util.extenstions.takeIfExists
import com.instabug.library.util.extenstions.takeUnlessExists
import org.json.JSONObject

const val NEWLINE_BYTE_VALUE = 0xA

/**
 * A base contract for atomic file operations with single responsibility.
 */
fun interface FileOperation<Input, Output> {
    /**
     * Executes the operation on the given input and returns output as specified.
     */
    operator fun invoke(input: Input): Output
}

// region Scope Ops
/**
 * A scope operation that allows for chaining two [FileOperation]s using the same input directory.
 * @constructor Takes 2 operations, the oldOperation is the one to execute first then the newOperation.
 */
class ContinueWithOp<SIn : Directory, NOpOut>(
    val oldOperation: FileOperation<SIn, *>,
    val newOperation: FileOperation<SIn, NOpOut>
) : FileOperation<SIn, NOpOut> {
    override fun invoke(input: SIn): NOpOut = run { oldOperation(input); newOperation(input) }

}

/**
 * A scope operation that selects a directory from the parent input that matches the needs
 * of the wrapped operation's input.
 * Using it multiple times acts as a Bottom-Walker in the file directory tree.
 * @constructor Takes a [DirectorySelector] with parent-type input and operation input-type output
 * and a [FileOperation] to be executed with the selector's output.
 */
class OperateOnDirectory<SIn : Directory, OpIn : Directory, OpOut>(
    private val directorySelector: DirectorySelector<SIn, OpIn>,
    private val operation: FileOperation<OpIn, OpOut>
) : FileOperation<SIn, OpOut?> {
    override fun invoke(input: SIn): OpOut? = runCatching {
        "[File Op] Operating on directory from parent $input".logDWithThreadName()
        input.let(directorySelector::invoke)
            ?.also { selected -> "[File Op] Selected directory $selected for operations".logVWithThreadName() }
            ?.let(operation::invoke)
            ?: run { "[File Op] Directory selector produced null or operation result is null".logDWithThreadName(); null }
    }.getOrLogAndReport(null, "[File Op] Error while operating on directory".prefixedWithT())
}

/**
 * A scope operation that selects a span of a multi-spans storage system [SpansDirectory].
 * @constructor Takes a [SpanSelector] with [SpansDirectory]-type input and operation input-type output
 * and a [FileOperation] to be executed with the selector's output.
 */
class OperateOnSpan<In : SpansDirectory<SpanDir>, SpanDir : Directory, Out>(
    private val spanSelector: SpanSelector<In, SpanDir>,
    private val operation: FileOperation<SpanDir, Out>
) : FileOperation<In, Out?> {
    override fun invoke(input: In): Out? = runCatching {
        "[File Op] Operating on a span from parent $input".logDWithThreadName()
        input.let(spanSelector::invoke)
            ?.also { selected -> "[File Op] Selected span directory $selected for operations".logVWithThreadName() }
            ?.let(operation::invoke)
            ?: run { "[File Op] Span selector produced null or operation result is null".logDWithThreadName(); null }
    }.getOrLogAndReport(null, "[File Op] Error while operating on span".prefixedWithT())
}

/**
 * A scope operation that selects a multiple spans of a multi-spans storage system [SpansDirectory].
 * @constructor Takes a [SpanSelector] with [SpansDirectory]-type input and operation input-type-list output
 * and a [FileOperation] to be executed on each selector output.
 */
class OperateOnMultiSpans<In : SpansDirectory<SpanDir>, SpanDir : Directory, Out>(
    private val spansSelector: MultiSpanSelector<In, SpanDir>,
    private val operation: FileOperation<SpanDir, Out>
) : FileOperation<In, List<Out>> {
    override fun invoke(input: In): List<Out> = runCatching {
        "[File Op] Operating on multi spans from parent $input".logDWithThreadName()
        input.let(spansSelector::invoke)
            .also { selected -> "[File Op] Selected ${selected.size} spans for operations".logVWithThreadName() }
            .map(operation::invoke)
    }.getOrLogAndReport(
        emptyList(),
        "[File Op] Error while operating on multi spans".prefixedWithT()
    )
}

// endregion

// region Basic Ops
class MakeDirectory<In : Directory> : FileOperation<In, Unit> {
    override fun invoke(input: In) {
        runCatching {
            "[File Op] Making directory $input.".logDWithThreadName()
            input.takeUnlessExists()?.mkdirDefensive()
                ?: "[File Op] Directory already exists.".logDWithThreadName()
        }.runOrLogAndReport("[File Op] Error while making directory.".prefixedWithT())
    }
}

class MakeDirectoryWithAncestors<In : Directory> : FileOperation<In, Unit> {
    override fun invoke(input: In) {
        runCatching {
            "[File Op] Making directory $input with ancestors.".logDWithThreadName()
            input.takeUnlessExists()?.mkdirsDefensive()
                ?: "[File Op] Directory already exists.".logDWithThreadName()
        }.runOrLogAndReport("[File Op] Error while making directory with ancestors.".prefixedWithT())
    }
}

class DeleteDirectory<In : Directory> : FileOperation<In, Unit> {
    override fun invoke(input: In) {
        runCatching {
            "[File Op] Deleting directory $input".logDWithThreadName()
            input.takeIfExists()?.deleteRecursivelyDefensive()
                ?: "[File Op] Directory doesn't exist (already deleted)".logDWithThreadName()
        }.runOrLogAndReport("[File Op] Error while deleting directory.".prefixedWithT())
    }
}

class CreateNewFile<C : Directory>(
    private val fileSelector: FileSelector<C>
) : FileOperation<C, Unit> {
    override fun invoke(input: C) {
        runCatching {
            "[File Op] Creating new file in parent directory $input".logDWithThreadName()
            input.let(fileSelector::invoke)
                ?.also { file -> "[File Op] Selected file $file for operations".logVWithThreadName() }
                ?.takeUnlessExists()
                ?.createNewFileDefensive()
                ?: "[File Op] Selected file already exists".logDWithThreadName()
        }.runOrLogAndReport("[File Op] Error while creating new file.".prefixedWithT())
    }
}

class DeleteFile<C : Directory>(
    private val fileSelector: FileSelector<C>
) : FileOperation<C, Unit> {
    override fun invoke(input: C) {
        runCatching {
            "[File Op] Deleting file in parent directory $input".logDWithThreadName()
            input.let(fileSelector::invoke)
                ?.takeIfExists()
                ?.deleteDefensive()
        }.runOrLogAndReport("[File Op] Error while deleting file.".prefixedWithT())
    }
}

class DeleteOldestFilesOnLimit<C : Directory>(
    private val limit: Int
) : FileOperation<C, Unit> {
    override fun invoke(input: C) {
        runCatching {
            "[File Op] Deleting oldest files on limit $limit in parent directory $input".logDWithThreadName()
            val existingFiles = input.listFiles()?.sortedBy { it.lastModified() }.orEmpty()
            val limitDelta = existingFiles.size - limit
            if (limitDelta < 0) return
            for (index in 0 until limitDelta) existingFiles[index].deleteDefensive()
        }.runOrLogAndReport("[File Op] Error while deleting oldest files on limit.".prefixedWithT())
    }

}

class OverwriteFile<In : Directory>(
    private val fileSelector: FileSelector<In>,
    private val cacheable: Cacheable
) : FileOperation<In, Unit> {
    override fun invoke(input: In) {
        runCatching {
            "[File Op] Overwriting file in parent directory $input".logDWithThreadName()
            input.let(fileSelector::invoke)
                ?.also { file -> "[File Op] Selected $file for operations".logVWithThreadName() }
                ?.takeIfExists()
                ?.outputStream()
                ?.use { stream ->
                    cacheable.toJson()?.toString()?.toByteArray()?.also(stream::write)
                } ?: "[File Op] Selected file doesn't exist".logDWithThreadName()
        }.runOrLogAndReport("[File Op] Error while overwriting file.".prefixedWithT())
    }
}

class WriteJSONToFile<In : Directory>(
    private val fileSelector: FileSelector<In>,
    private val json: JSONObject
) : FileOperation<In, Unit> {
    override fun invoke(input: In) {
        runCatching {
            "[File Op] Writing JSON to file in parent directory $input".logDWithThreadName()
            input.let(fileSelector::invoke)
                ?.also { file -> "[File Op] Selected $file for operations".logVWithThreadName() }
                ?.takeIfExists()
                ?.outputStream()
                ?.use { stream ->
                    json.toString().toByteArray().also(stream::write).also { stream.flush() }
                } ?: "[File Op] Selected file doesn't exist".logDWithThreadName()
        }.runOrLogAndReport("[File Op] Error while overwriting file.".prefixedWithT())
    }
}

class ReadJSONFromFile<In : Directory, Out>(
    private val fileSelector: FileSelector<In>,
    private val aggregator: DataAggregator<Out>
) : FileOperation<In, Out> {
    override fun invoke(input: In): Out {
        val emptyAggregate = aggregator.aggregate()
        return runCatching {
            "[File Op] Reading JSON from file in parent directory $input".logDWithThreadName()
            input.let(fileSelector::invoke)
                ?.also { file -> "[File Op] Selected $file for operations".logVWithThreadName() }
                ?.takeIfExists()
                ?.bufferedReader()
                ?.readJSONLines(aggregator::add)
                ?: "[File Op] Selected file for operations doesn't exist".logDWithThreadName()
            aggregator.aggregate()
        }.getOrLogAndReport(
            emptyAggregate,
            "[File Op] Error while Reading JSON from file".prefixedWithT()
        )
    }
}

// endregion