package com.instabug.library.encryption.iv

import android.os.Build
import android.security.keystore.KeyGenParameterSpec
import android.security.keystore.KeyProperties
import android.util.Base64.DEFAULT
import android.util.Base64.decode
import android.util.Base64.encodeToString
import androidx.annotation.RequiresApi
import com.instabug.library.Constants
import com.instabug.library.diagnostics.IBGDiagnostics
import com.instabug.library.encryption.EncryptionPreferences
import com.instabug.library.encryption.iv.SecureIVKeyStoreManager.getIV
import com.instabug.library.internal.contentprovider.InstabugApplicationProvider
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.nullRetryLazy
import java.io.ByteArrayInputStream
import java.io.ByteArrayOutputStream
import java.security.KeyPairGenerator
import java.security.KeyStore
import java.security.PrivateKey
import java.security.PublicKey
import java.security.SecureRandom
import javax.crypto.Cipher
import javax.crypto.CipherInputStream
import javax.crypto.CipherOutputStream

/**
 * Manages the generation, storage, and usage of an RSA key pair in the Android Keystore
 * for encrypting and decrypting IVs (Initialization Vectors).
 *
 * ## Compatibility
 * - Requires API 23+ (Android M and above).
 * - Uses KeyGenParameterSpec for key generation, which does not support custom key expiration dates.
 *
 * ## Key Expiration
 * - The generated RSA key does not have an explicit expiration date.
 * - The key remains valid until the app is uninstalled, data is cleared, or the key is explicitly deleted.
 * - If you need key expiration, implement your own logic to track and rotate keys as needed.
 *
 * ## Security
 * - The key is stored securely in the Android Keystore and is not accessible outside the device.
 * - Encryption uses RSA with OAEP padding and SHA-256/SHA-512 digests for strong security.
 *
 * ## Usage
 * - Call [getIV] to retrieve the decrypted IV, generating and storing it if necessary.
 * - The IV is encrypted with the RSA public key and stored in preferences.
 *
 * ## Changes
 * - Updated to use only KeyGenParameterSpec for API 23+ (removed deprecated KeyPairGeneratorSpec).
 * - Documented the lack of key expiration and how to handle it if needed.
 * - Improved code readability and error handling.
 *
 * ## Suggestions for further improvement
 * - Remove unnecessary imports (e.g., SecureIVKeyStoreManager.getIV import inside the object).
 * - Define IV size as a constant for clarity.
 * - Consider using requireNotNull for keystore if you want to fail fast.
 * - Use extension functions for repetitive code (e.g., encoding/decoding).
 * - Ensure thread safety if high concurrency is expected (Mutex, etc.).
 * - Add unit tests for all edge cases.
 */
@RequiresApi(Build.VERSION_CODES.M)
object SecureIVKeyStoreManager {
    private const val RSA_MODE = "RSA/ECB/OAEPWithSHA-256AndMGF1Padding"
    private const val RSA_KEY_ALIAS = "iv_rsa_keys"
    private const val ANDROID_KEYSTORE = "AndroidKeyStore"
    private const val IV_SIZE = 12

    private val keystore: KeyStore? by nullRetryLazy {
        runCatching {
            KeyStore.getInstance(ANDROID_KEYSTORE).apply { load(null) }
        }.reportAndLogOnFailure("Error while instantiating keystore").getOrNull()
    }

    private fun hasRSAKey(): Boolean = keystore?.containsAlias(RSA_KEY_ALIAS) == true

    /**
     * Retrieves the decrypted IV, generating and storing it if necessary.
     * @return The decrypted IV as a ByteArray.
     */
    @JvmStatic
    fun getIV(): ByteArray = EncryptionPreferences.getEncryptionIV()
        .takeIf { it.isNotBlank() }
        ?.let(::rsaDecrypt)
        ?: generateIVAndStoreIt()

    /**
     * Generates a new IV, encrypts it, and stores it in preferences.
     * @return The generated IV as a ByteArray.
     */
    @Synchronized
    private fun generateIVAndStoreIt(): ByteArray {
        if (keystore != null && keystore?.containsAlias(RSA_KEY_ALIAS) != true) generateRSAKeys()
        val iv = ByteArray(IV_SIZE).also(SecureRandom()::nextBytes)
        val encryptedIV = rsaEncrypt(iv)
        EncryptionPreferences.saveEncryptionIV(encryptedIV ?: "")
        return iv
    }

    /**
     * Generates an RSA key pair and stores it in the Android Keystore.
     */
    private fun generateRSAKeys() =
        InstabugApplicationProvider.getInstance()?.application
            ?.runCatching {
                val spec = parameterSpec()
                KeyPairGenerator.getInstance(KeyProperties.KEY_ALGORITHM_RSA, ANDROID_KEYSTORE)
                    .apply {
                        initialize(spec)
                        generateKeyPair()
                    }
            }
            ?.reportAndLogOnFailure("Error while generating RSA keys")
            ?.getOrNull()

    private fun parameterSpec() = KeyGenParameterSpec.Builder(
        RSA_KEY_ALIAS, KeyProperties.PURPOSE_ENCRYPT or KeyProperties.PURPOSE_DECRYPT
    ).setDigests(KeyProperties.DIGEST_SHA256, KeyProperties.DIGEST_SHA512)
        .setEncryptionPaddings(KeyProperties.ENCRYPTION_PADDING_RSA_OAEP)
        .setKeySize(2048).build()


    /**
     * Encrypts the given secret using the RSA public key from the keystore.
     * @param secret The data to encrypt.
     * @return The encrypted data as a Base64-encoded string, or null if encryption fails.
     */
    private fun rsaEncrypt(secret: ByteArray): String? = keystore
        ?.takeIf { hasRSAKey() }
        ?.run { getPublicKey() }
        ?.runCatching { encrypt(getInputCipher(), secret) }
        ?.reportAndLogOnFailure("Error while encrypting IV using RSA")
        ?.getOrNull()

    private fun encrypt(inputCipher: Cipher?, secret: ByteArray) = ByteArrayOutputStream()
        .use { outputStream ->
            CipherOutputStream(outputStream, inputCipher).use { cipherOutputStream ->
                cipherOutputStream.write(secret)
            }
            encodeToString(outputStream.toByteArray(), DEFAULT)
        }


    private fun KeyStore.getPublicKey() =
        (getEntry(RSA_KEY_ALIAS, null) as? KeyStore.PrivateKeyEntry)
            ?.certificate
            ?.publicKey

    private fun PublicKey.getInputCipher() = Cipher.getInstance(RSA_MODE).also { cipher ->
        cipher.init(Cipher.ENCRYPT_MODE, this)
    }


    /**
     * Decrypts the given Base64-encoded string using the RSA private key from the keystore.
     * @param encrypted The encrypted data as a Base64-encoded string.
     * @return The decrypted data as a ByteArray, or null if decryption fails.
     */
    private fun rsaDecrypt(encrypted: String): ByteArray? = keystore
        ?.takeIf { hasRSAKey() }
        ?.run { getPrivateKey() }
        ?.runCatching { decrypt(getOutputCipher(), encrypted) }
        ?.reportAndLogOnFailure("Error while decrypting encryption IV using RSA")
        ?.getOrNull()

    private fun decrypt(outputCipher: Cipher?, encrypted: String): ByteArray {
        val encryptedBytes = decode(encrypted, DEFAULT)
        return CipherInputStream(
            ByteArrayInputStream(encryptedBytes),
            outputCipher
        ).use { cipherInputStream ->
            cipherInputStream.readBytes()
        }
    }

    private fun KeyStore.getPrivateKey() =
        (getEntry(RSA_KEY_ALIAS, null) as? KeyStore.PrivateKeyEntry)?.privateKey

    private fun PrivateKey.getOutputCipher() = Cipher.getInstance(RSA_MODE).also { cipher ->
        cipher.init(Cipher.DECRYPT_MODE, this)
    }

    private fun <T> Result<T>.reportAndLogOnFailure(message: String) = onFailure { throwable ->
        InstabugSDKLogger.e(Constants.LOG_TAG, message, throwable)
        IBGDiagnostics.reportNonFatal(throwable, message)
    }
}
