package com.instabug.library.encryption

import android.os.Build
import android.security.KeyPairGeneratorSpec
import android.util.Base64.*
import androidx.annotation.RequiresApi
import com.instabug.library.Constants
import com.instabug.library.diagnostics.IBGDiagnostics
import com.instabug.library.internal.contentprovider.InstabugApplicationProvider
import com.instabug.library.util.InstabugSDKLogger
import java.io.ByteArrayInputStream
import java.io.ByteArrayOutputStream
import java.math.BigInteger
import java.security.Key
import java.security.KeyPairGenerator
import java.security.KeyStore
import java.security.SecureRandom
import java.util.*
import javax.crypto.Cipher
import javax.crypto.CipherInputStream
import javax.crypto.CipherOutputStream
import javax.crypto.spec.SecretKeySpec
import javax.security.auth.x500.X500Principal

object Post18KeyGenerator {

    private const val RSA_MODE = "RSA/ECB/PKCS1Padding"
    private const val RSA_KEY_ALIAS = "rsa_keys"
    private const val AES_ALGORITHM = "AES"
    private const val androidKeyStore = "AndroidKeyStore"

    private var keystore: KeyStore? = null


    init {
        if (keystore == null) {
            try {
                keystore = KeyStore.getInstance(androidKeyStore)
                keystore?.load(null)
            } catch (e: Exception) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Error while instantiating keystore")
                IBGDiagnostics.reportNonFatal(e, "Error while instantiating keystore")
                keystore = null
            }
        }
    }

    @RequiresApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
    private fun generateRSAKeys() {


        InstabugApplicationProvider.getInstance()?.application?.let {
            try {
                val start: Calendar = Calendar.getInstance()
                val end: Calendar = Calendar.getInstance()
                //Keys will expire in this date
                end.add(Calendar.YEAR, 30)
                val spec: KeyPairGeneratorSpec = KeyPairGeneratorSpec.Builder(it)
                        .setAlias(RSA_KEY_ALIAS)
                        .setSubject(X500Principal("CN=${RSA_KEY_ALIAS}"))
                        .setSerialNumber(BigInteger.ONE)
                        .setStartDate(start.time)
                        .setEndDate(end.time)
                        .build()
                val generator: KeyPairGenerator = KeyPairGenerator.getInstance("RSA", androidKeyStore)

                generator.initialize(spec)
                generator.generateKeyPair()

            } catch (e: Exception) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Error while generating RSA keys")
                IBGDiagnostics.reportNonFatal(e, "Error while generating RSA keys")
            }
        }
    }

    private fun rsaEncrypt(secret: ByteArray): String? {
        var privateKeyEntry: KeyStore.PrivateKeyEntry
        if (keystore != null) {
            try {
                privateKeyEntry = keystore!!.getEntry(RSA_KEY_ALIAS, null) as KeyStore.PrivateKeyEntry
                val inputCipher: Cipher = Cipher.getInstance(RSA_MODE, "AndroidOpenSSL")
                inputCipher.init(Cipher.ENCRYPT_MODE, privateKeyEntry.certificate.publicKey)
                val outputStream = ByteArrayOutputStream()
                val cipherOutputStream = CipherOutputStream(outputStream, inputCipher)
                cipherOutputStream.write(secret)
                cipherOutputStream.close()
                val encrypted: ByteArray = outputStream.toByteArray()
                return encodeToString(encrypted, DEFAULT)

            } catch (exception: java.lang.Exception) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Error while encrypting key using RSA")
                IBGDiagnostics.reportNonFatal(exception, "Error while encrypting key using RSA")
            } catch (oom: OutOfMemoryError) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "OOM while encrypting key using RSA")
                IBGDiagnostics.reportNonFatal(oom, "OOM while encrypting key using RSA")
            }
        }
        return ""
    }

    private fun rsaDecrypt(encrypted: String): ByteArray? {
        if (keystore != null) {
            return try {
                val encryptedBytes: ByteArray = decode(encrypted, DEFAULT)
                val privateKeyEntry = keystore?.getEntry(RSA_KEY_ALIAS, null) as KeyStore.PrivateKeyEntry
                val output: Cipher = Cipher.getInstance(RSA_MODE, "AndroidOpenSSL")
                output.init(Cipher.DECRYPT_MODE, privateKeyEntry.privateKey)
                val cipherInputStream = CipherInputStream(
                        ByteArrayInputStream(encryptedBytes), output)
                val values: ArrayList<Byte> = ArrayList()
                var nextByte: Int
                while (cipherInputStream.read().also { nextByte = it } != -1) {
                    values.add(nextByte.toByte())
                }
                val bytes = ByteArray(values.size)
                for (i in bytes.indices) {
                    bytes[i] = values[i]
                }
                bytes
            } catch (e: java.lang.Exception) {
                InstabugSDKLogger.e(
                    Constants.LOG_TAG,
                    "Error while decrypting encryption key using RSA"
                )
                IBGDiagnostics.reportNonFatal(e, "Error while decrypting encryption key using RSA")
                null
            } catch (oom: OutOfMemoryError) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "OOM while decrypting key using RSA")
                IBGDiagnostics.reportNonFatal(oom, "OOM while decrypting encryption key using RSA")
                null
            }
        }
        return null;
    }

    @JvmStatic
    @RequiresApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
    fun getKeyPostApi18(): Key? {
        if (EncryptionPreferences.getEncryptionKey().isEmpty()) {
            generateKeyApi18()
        }

        val decryptedKeyBytes = rsaDecrypt(EncryptionPreferences.getEncryptionKey())
        return if (decryptedKeyBytes != null && decryptedKeyBytes.isNotEmpty()) {
            SecretKeySpec(decryptedKeyBytes, AES_ALGORITHM)
        } else {
            null
        }
    }

    @RequiresApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
    fun generateKeyApi18() {
        if (keystore != null && keystore?.containsAlias(RSA_KEY_ALIAS) != true) {
            generateRSAKeys()
        }

        val key = ByteArray(32)
        SecureRandom().nextBytes(key)

        val encryptedKey = rsaEncrypt(key)
        EncryptionPreferences.saveEncryptionKey(encryptedKey ?: "")
    }

}