package com.instabug.library.diagnostics.nonfatals.di;

import android.content.Context;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.Instabug;
import com.instabug.library.diagnostics.configuration.ConfigurationChangedHandler;
import com.instabug.library.diagnostics.diagnostics_db.DiagnosticsDbManager;
import com.instabug.library.diagnostics.mappers.RequestParameterResolver;
import com.instabug.library.diagnostics.nonfatals.NonFatalsConfigurationChangedHandler;
import com.instabug.library.diagnostics.nonfatals.NonFatalsManager;
import com.instabug.library.diagnostics.nonfatals.NonFatalsManagerImpl;
import com.instabug.library.diagnostics.nonfatals.cache.NonFatalCacheManager;
import com.instabug.library.diagnostics.nonfatals.cache.NonFatalCacheManagerImpl;
import com.instabug.library.diagnostics.nonfatals.cache.NonFatalsDBHelper;
import com.instabug.library.diagnostics.nonfatals.cache.NonFatalsDBHelperImpl;
import com.instabug.library.diagnostics.nonfatals.cache.OccurrencesDBHelper;
import com.instabug.library.diagnostics.nonfatals.cache.OccurrencesDBHelperImpl;
import com.instabug.library.diagnostics.nonfatals.networking.mapping.NonFatalMapper;
import com.instabug.library.diagnostics.nonfatals.networking.mapping.NonFatalMapperImpl;
import com.instabug.library.diagnostics.nonfatals.networking.mapping.NonFatalsRequestParamResolver;
import com.instabug.library.internal.resolver.NonFatalSettingResolver;
import com.instabug.library.util.threading.PoolProvider;

import org.json.JSONArray;

import java.lang.ref.WeakReference;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;

public final class ServiceLocator {

    private static final Map<String, WeakReference<Object>> objectsMap = new HashMap<>();

    @Nullable
    private static Object getIfAvailable(@NonNull String key) {
        if (objectsMap.containsKey(key)
                && objectsMap.get(key) != null && objectsMap.get(key).get() != null) {
            return objectsMap.get(key).get();
        }
        return null;
    }

    @Nullable
    public static synchronized Context getContext() {
        return Instabug.getApplicationContext();
    }

    @Nullable
    public synchronized static NonFatalCacheManager getNonFatalCacheManager() {
        String key = "NonFatalCacheManager";
        Object cacheManager = getIfAvailable(key);
        if (cacheManager == null && NonFatalSettingResolver.getInstance().getNonFatalsSettings() != null) {
            cacheManager = new NonFatalCacheManagerImpl(
                    getNonFatalsDBHelper(),
                    getOccurrencesDBHelper(),
                    NonFatalSettingResolver.getInstance().getNonFatalsSettings()
            );
            objectsMap.put(key, new WeakReference<>(cacheManager));
        }
        return (NonFatalCacheManager) cacheManager;
    }

    @Nullable
    public synchronized static NonFatalsManager getNonFatalsManager() {
        String key = "NonFatalsManager";
        Object nonFatalsManager = getIfAvailable(key);
        if (nonFatalsManager == null) {
            if (NonFatalSettingResolver.getInstance().getNonFatalsSettings() != null && getNonFatalCacheManager() != null) {
                nonFatalsManager = new NonFatalsManagerImpl(
                        getNonFatalCacheManager(),
                        NonFatalSettingResolver.getInstance().getNonFatalsSettings()
                );
                objectsMap.put(key, new WeakReference<>(nonFatalsManager));
            }
        }
        return (NonFatalsManager) nonFatalsManager;
    }

    public synchronized static DiagnosticsDbManager getDatabaseManager() {
        return DiagnosticsDbManager.getInstance();
    }

    public synchronized static NonFatalsDBHelper getNonFatalsDBHelper() {
        String key = "NonFatalsDBHelper";
        Object dbHelper = getIfAvailable(key);
        if (dbHelper == null) {
            dbHelper = new NonFatalsDBHelperImpl();
            objectsMap.put(key, new WeakReference<>(dbHelper));
        }
        return (NonFatalsDBHelper) dbHelper;
    }

    public synchronized static OccurrencesDBHelper getOccurrencesDBHelper() {
        String key = "OccurrencesDBHelper";
        Object dbHelper = getIfAvailable(key);
        if (dbHelper == null) {
            dbHelper = new OccurrencesDBHelperImpl();
            objectsMap.put(key, new WeakReference<>(dbHelper));
        }
        return (OccurrencesDBHelper) dbHelper;
    }

    public synchronized static NonFatalMapper getNonFatalMapper() {
        String key = "NonFatalMapper";
        Object nonFatalMapper = getIfAvailable(key);
        if (nonFatalMapper == null) {
            nonFatalMapper = new NonFatalMapperImpl();
            objectsMap.put(key, new WeakReference<>(nonFatalMapper));
        }
        return (NonFatalMapper) nonFatalMapper;
    }

    public synchronized static ConfigurationChangedHandler getNonFatalsConfigurationHandler() {
        String key = "ConfigurationHandler";
        Object configurationHandler = getIfAvailable(key);
        if (configurationHandler == null) {
            configurationHandler = new NonFatalsConfigurationChangedHandler();
            objectsMap.put(key, new WeakReference<>(configurationHandler));
        }
        return (ConfigurationChangedHandler) configurationHandler;
    }

    public synchronized static RequestParameterResolver<JSONArray> getNonFatalsRequestParamResolver() {
        String key = "NonFatalsRequestParamMapper";
        Object requestParameterMapper = getIfAvailable(key);
        if (requestParameterMapper == null) {
            requestParameterMapper = new NonFatalsRequestParamResolver();
            objectsMap.put(key, new WeakReference<>(requestParameterMapper));
        }
        return (NonFatalsRequestParamResolver) requestParameterMapper;
    }

    public static synchronized ExecutorService getNonFatalsThreadExecutor() {
        return PoolProvider.getReturnableSingleThreadExecutor("ibg-non-fatal-executor");
    }

    public synchronized static void reset() {
        objectsMap.clear();
    }
}
