package com.instabug.library.diagnostics.nonfatals.cache;


import static com.instabug.library.diagnostics.diagnostics_db.NonFatalEntry.COLUMN_EXCEPTION_TYPE;
import static com.instabug.library.diagnostics.diagnostics_db.NonFatalEntry.COLUMN_ID;
import static com.instabug.library.diagnostics.diagnostics_db.NonFatalEntry.COLUMN_MESSAGE;
import static com.instabug.library.diagnostics.diagnostics_db.NonFatalEntry.COLUMN_PRIORITY;
import static com.instabug.library.diagnostics.diagnostics_db.NonFatalEntry.COLUMN_STACKTRACE;
import static com.instabug.library.diagnostics.diagnostics_db.NonFatalEntry.COLUMN_STACKTRACE_DECLARING_CLASS;
import static com.instabug.library.diagnostics.diagnostics_db.NonFatalEntry.COLUMN_STACKTRACE_FILE_NAME;
import static com.instabug.library.diagnostics.diagnostics_db.NonFatalEntry.COLUMN_STACKTRACE_LINE_NUMBER;
import static com.instabug.library.diagnostics.diagnostics_db.NonFatalEntry.COLUMN_STACKTRACE_METHOD_NAME;
import static com.instabug.library.diagnostics.diagnostics_db.NonFatalEntry.TABLE_NAME;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.Constants;
import com.instabug.library.diagnostics.diagnostics_db.DiagnosticsDbManager;
import com.instabug.library.diagnostics.nonfatals.di.ServiceLocator;
import com.instabug.library.diagnostics.nonfatals.model.NonFatal;
import com.instabug.library.internal.storage.cache.dbv2.IBGContentValues;
import com.instabug.library.internal.storage.cache.dbv2.IBGCursor;
import com.instabug.library.internal.storage.cache.dbv2.IBGWhereArg;
import com.instabug.library.util.InstabugSDKLogger;

import org.jetbrains.annotations.NotNull;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

/**
 * Created by Shannan on 31/08/2021.
 */
public final class NonFatalsDBHelperImpl implements NonFatalsDBHelper {


    @Nullable
    private final DiagnosticsDbManager databaseManager = ServiceLocator.getDatabaseManager();

    @NotNull
    public static NonFatalsDBHelperImpl getInstance() {
        return new NonFatalsDBHelperImpl();
    }

    @Override
    public void trimToLimit(@Nullable List<Long> ids) {
        if (databaseManager != null && ids != null) {
            try {
                databaseManager.beginTransaction();
                for (Long id : ids) {
                    IBGWhereArg ibgWhereArg = new IBGWhereArg(id.toString(), ColumnsTransitiveStates.NON_FATAL_ID);
                    databaseManager.delete(TABLE_NAME, COLUMN_ID + " = ?", Collections.singletonList(ibgWhereArg));
                }
                databaseManager.endTransaction();

            } catch (Exception e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while trimming non-fatal table", e);
            }
        }
    }

    @Override
    public long insertNonFatal(@NonNull NonFatal nonFatal) {
        if (databaseManager != null) {
            try {
                IBGContentValues contentValues = new IBGContentValues();
                if (nonFatal.getId() != -1) {
                    contentValues.put(COLUMN_ID, nonFatal.getId(), ColumnsTransitiveStates.NON_FATAL_ID);
                }
                if (nonFatal.getExceptionType() != null)
                    contentValues.put(COLUMN_EXCEPTION_TYPE, nonFatal.getExceptionType(), ColumnsTransitiveStates.EXCEPTION_TYPE);
                if (nonFatal.getDeclaringClass() != null)
                    contentValues.put(COLUMN_STACKTRACE_DECLARING_CLASS, nonFatal.getDeclaringClass(), ColumnsTransitiveStates.STACKTRACE_DECLARING_CLASS);
                if (nonFatal.getFileName() != null)
                    contentValues.put(COLUMN_STACKTRACE_FILE_NAME, nonFatal.getFileName(), ColumnsTransitiveStates.STACKTRACE_FILE_NAME);
                if (nonFatal.getMethodName() != null)
                    contentValues.put(COLUMN_STACKTRACE_METHOD_NAME, nonFatal.getMethodName(), ColumnsTransitiveStates.STACKTRACE_METHOD_NAME);
                contentValues.put(COLUMN_STACKTRACE_LINE_NUMBER, nonFatal.getLineNumber(), ColumnsTransitiveStates.STACKTRACE_LINE_NUMBER);
                contentValues.put(COLUMN_MESSAGE, nonFatal.getMessage(), ColumnsTransitiveStates.MESSAGE);
                if (nonFatal.getStackTrace() != null) {
                    contentValues.put(COLUMN_STACKTRACE, nonFatal.getStackTrace(), ColumnsTransitiveStates.STACKTRACE);
                }

                contentValues.put(COLUMN_PRIORITY, nonFatal.getPriority(), ColumnsTransitiveStates.PRIORITY);

                return databaseManager.insertWithOnConflictReplace(TABLE_NAME, null, contentValues);
            } catch (Exception e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while inserting non-fatal", e);
            }
        }
        return -1;
    }

    @Override
    public long getNonFatalId(NonFatal nonFatal) {
        if (nonFatal != null && databaseManager != null) {

            String selectionClause = COLUMN_EXCEPTION_TYPE + " = ?"
                    + " and " + COLUMN_STACKTRACE_DECLARING_CLASS + " = ?"
                    + " and " + COLUMN_STACKTRACE_FILE_NAME + " = ?"
                    + " and " + COLUMN_STACKTRACE_METHOD_NAME + " = ?"
                    + " and " + COLUMN_STACKTRACE_LINE_NUMBER + " = ?";

            List<IBGWhereArg> selectionArgs = new ArrayList<>();
            if (nonFatal.getExceptionType() != null)
                selectionArgs.add(new IBGWhereArg(nonFatal.getExceptionType(), ColumnsTransitiveStates.EXCEPTION_TYPE));
            if (nonFatal.getDeclaringClass() != null)
                selectionArgs.add(new IBGWhereArg(nonFatal.getDeclaringClass(), ColumnsTransitiveStates.STACKTRACE_DECLARING_CLASS));
            if (nonFatal.getFileName() != null)
                selectionArgs.add(new IBGWhereArg(nonFatal.getFileName(), ColumnsTransitiveStates.STACKTRACE_FILE_NAME));
            if (nonFatal.getMethodName() != null)
                selectionArgs.add(new IBGWhereArg(nonFatal.getMethodName(), ColumnsTransitiveStates.STACKTRACE_METHOD_NAME));
            if (nonFatal.getLineNumber() != 0)
                selectionArgs.add(new IBGWhereArg(String.valueOf(nonFatal.getLineNumber()), ColumnsTransitiveStates.STACKTRACE_LINE_NUMBER));


            IBGCursor cursor = null;
            try {
                cursor = databaseManager.query(TABLE_NAME, new String[]{COLUMN_ID}, selectionClause, selectionArgs, null, null, null);
                if (cursor != null && cursor.moveToFirst()) {
                    return cursor.getLong(cursor.getColumnIndex(COLUMN_ID));
                }
            } catch (Exception e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while retrieving non-fatal id", e);
            } finally {
                if (cursor != null) {
                    cursor.close();
                }
            }
        }
        return -1;
    }

    @Override
    public void clearAll() {
        if (databaseManager != null) {
            try {
                databaseManager.delete(TABLE_NAME, null, null);
            } catch (Exception e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while deleting non-fatals", e);
            }
        }
    }

    @Override
    @Nullable
    public List<Long> getEligibleToBeDeleted(int nonFatalsMaxCount) {
        if (databaseManager != null) {
            List<NonFatal> nonFatals = getAllNonFatals();
            if (nonFatals.size() > nonFatalsMaxCount) {
                int extraNonFatalsSize = nonFatals.size() - nonFatalsMaxCount;
                List<Long> eligibleToBeDeleted = new ArrayList<>();
                Iterator<NonFatal> iterator = nonFatals.iterator();
                while (iterator.hasNext() && eligibleToBeDeleted.size() < extraNonFatalsSize) {
                    NonFatal nonFatal = iterator.next();
                    if (nonFatal.getPriority() != NonFatal.Priority.HIGH) {
                        eligibleToBeDeleted.add(nonFatal.getId());
                        iterator.remove();
                    }
                }

                int index = 0;
                while (eligibleToBeDeleted.size() < extraNonFatalsSize) {
                    eligibleToBeDeleted.add(nonFatals.get(index).getId());
                    index++;
                }

                return eligibleToBeDeleted;
            }
        }
        return null;
    }

    @NonNull
    @Override
    public List<NonFatal> getAllNonFatals() {
        List<NonFatal> nonFatals = new ArrayList<>();
        if (databaseManager != null) {
            IBGCursor cursor = null;
            try {
                cursor = databaseManager.query(TABLE_NAME, null, null, null, null, null, null);
                if (cursor != null && cursor.moveToFirst()) {
                    do {
                        NonFatal nonFatal = new NonFatal();
                        nonFatal.setId(cursor.getLong(cursor.getColumnIndex(COLUMN_ID)));
                        nonFatal.setExceptionType(cursor.getString(cursor.getColumnIndex(COLUMN_EXCEPTION_TYPE)));
                        nonFatal.setDeclaringClass(cursor.getString(cursor.getColumnIndex(COLUMN_STACKTRACE_DECLARING_CLASS)));
                        nonFatal.setFileName(cursor.getString(cursor.getColumnIndex(COLUMN_STACKTRACE_FILE_NAME)));
                        nonFatal.setMethodName(cursor.getString(cursor.getColumnIndex(COLUMN_STACKTRACE_METHOD_NAME)));
                        nonFatal.setLineNumber(cursor.getInt(cursor.getColumnIndex(COLUMN_STACKTRACE_LINE_NUMBER)));
                        nonFatal.setMessage(cursor.getString(cursor.getColumnIndex(COLUMN_MESSAGE)));
                        nonFatal.setStackTrace(cursor.getString(cursor.getColumnIndex(COLUMN_STACKTRACE)));
                        nonFatal.setPriority(cursor.getInt(cursor.getColumnIndex(COLUMN_PRIORITY)));
                        nonFatals.add(nonFatal);
                    } while (cursor.moveToNext());
                }
            } catch (Exception e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while retrieving non-fatals", e);
            } finally {
                if (cursor != null) {
                    try {
                        cursor.close();
                    } catch (Exception e) {
                        InstabugSDKLogger.e(Constants.LOG_TAG, "Cursor not closed", e);
                    }
                }
            }
        }

        return nonFatals;
    }

    @Override
    public void deleteNonFatal(long id) {
        if (databaseManager != null) {
            try {
                String whereClause = COLUMN_ID + " = ?";
                List<IBGWhereArg> args = new ArrayList<>();
                args.add(new IBGWhereArg(String.valueOf(id), ColumnsTransitiveStates.NON_FATAL_ID));
                databaseManager.delete(TABLE_NAME, whereClause, args);
            } catch (Exception e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while deleting non-fatals", e);
            }
        }
    }
}
