package com.instabug.library.diagnostics.customtraces.di

import com.instabug.library.diagnostics.configuration.ConfigurationChangedHandler
import com.instabug.library.diagnostics.customtraces.CustomTracesManager
import com.instabug.library.diagnostics.customtraces.CustomTracesManagerImpl
import com.instabug.library.diagnostics.customtraces.cache.*
import com.instabug.library.diagnostics.customtraces.network.CustomTracesMapper
import com.instabug.library.diagnostics.customtraces.network.CustomTracesMapperImpl
import com.instabug.library.diagnostics.customtraces.network.CustomTracesRequestParamResolver
import com.instabug.library.diagnostics.customtraces.settings.TracesConfigurationHandler
import com.instabug.library.diagnostics.mappers.RequestParameterResolver
import com.instabug.library.util.threading.PoolProvider
import com.instabug.library.util.threading.ReturnableSingleThreadExecutor
import org.json.JSONObject
import java.lang.ref.WeakReference
import java.util.concurrent.Executor
import java.util.concurrent.ThreadPoolExecutor

object CustomTracesServiceLocator {
    private val objectsMap = mutableMapOf<String, WeakReference<Any>>()

    private inline fun <reified T> getIfAvailable(key: String): T? {
        return objectsMap[key]?.get() as? T
    }

    @Synchronized
    fun getTracesConfigurationHandler(): ConfigurationChangedHandler {
        return "ConfigurationHandler".let { key ->
            getIfAvailable<ConfigurationChangedHandler>(key)
                ?: TracesConfigurationHandler.also { handler ->
                    objectsMap[key] = WeakReference(handler)
                }
        }
    }

    @Synchronized
    fun getTracesDBHelper(): CustomTracesDBHelper {
        return "TracesDBHelper".let { key ->
            getIfAvailable<CustomTracesDBHelper>(key)
                ?: CustomTracesDBHelperImpl().also { handler ->
                    objectsMap[key] = WeakReference(handler)
                }
        }
    }

    @Synchronized
    fun getAttributesDBHelper(): AttributesDBHelper {
        return "AttributesDBHelper".let { key ->
            getIfAvailable<AttributesDBHelper>(key)
                ?: AttributesDBHelperImpl().also { handler ->
                    objectsMap[key] = WeakReference(handler)
                }
        }

    }

    @Synchronized
    fun getCacheManager(): CustomTracesCacheManager =
        "CacheManager".let { key ->
            getIfAvailable<CustomTracesCacheManager>(key)
                ?: CustomTracesCacheManagerImpl().also { cacheManager ->
                    objectsMap[key] = WeakReference(cacheManager)
                }
        }


    @JvmStatic
    @Synchronized
    fun getCustomTracesManager(): CustomTracesManager =
        "TracesManager".let { key ->
            getIfAvailable<CustomTracesManager>(key)
                ?: CustomTracesManagerImpl().also { tracesManager ->
                    objectsMap[key] = WeakReference(tracesManager)
                }
        }

    @Synchronized
    fun getCustomTracesExecutor(): ThreadPoolExecutor = PoolProvider.getInstance().ioExecutor

    @Synchronized
    fun getCustomTracesMapper(): CustomTracesMapper =
        "TracesMapper".let { key ->
            getIfAvailable<CustomTracesMapper>(key)
                ?: CustomTracesMapperImpl().also { mapper ->
                    objectsMap[key] = WeakReference(mapper)
                }
        }

    @Synchronized
    fun getCustomTracesRequestParamResolver(): RequestParameterResolver<JSONObject> =
        "RequestParamResolver".let { key ->
            getIfAvailable<CustomTracesRequestParamResolver>(key)
                ?: CustomTracesRequestParamResolver().also { resolver ->
                    objectsMap[key] = WeakReference(resolver)
                }
        }
}