package com.instabug.library.datahub

import com.instabug.library.IBGFeature
import com.instabug.library.core.InstabugCore
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent
import com.instabug.library.internal.filestore.ActiveCurrentSpanSelector
import com.instabug.library.internal.filestore.Directory
import com.instabug.library.internal.filestore.DirectorySelector
import com.instabug.library.internal.filestore.JSONArrayDataAggregator
import com.instabug.library.internal.filestore.SpanSelector
import com.instabug.library.model.NetworkLog
import com.instabug.library.model.State
import com.instabug.library.util.LimitConstraintApplier
import com.instabug.library.util.extenstions.getOrLogAndReport
import com.instabug.library.util.extenstions.logDWithThreadName
import com.instabug.library.util.extenstions.logVWithThreadName
import com.instabug.library.util.extenstions.runOrLogAndReport
import com.instabug.library.util.threading.OrderedExecutorService
import org.json.JSONArray
import java.util.concurrent.Future

private const val NETWORK_LOGS_STORE_LIMIT = 100

class NetworkLogsHubDataStore(
    executor: OrderedExecutorService,
    batcher: LogsBatcher
) : HubBatchedDataStore<NetworkLog>(executor, batcher), ReportContributor, HubEventsListener {

    override val storeDirectorySelector: DirectorySelector<HubLaunchDirectory, Directory> =
        DirectorySelector { launchDirectory -> launchDirectory.networkLogsDirectory }

    override val executionQueue: String
        get() = HubExecutionQueues.NetworkLogsStore

    override val logsPrefix: String = "Network logs"

    override fun blockingInit() {
        if (!isFeatureEnabled()) return
        super.blockingInit()
    }

    override fun clear() {
        // No-Op: Network logs are not designed to be cleared by any triggers.
    }

    override fun onNewEvent(event: IBGSdkCoreEvent) {
        executeOnStoreQueue {
            "[Hub] $logsPrefix data store received event $event".logDWithThreadName()
            when (event) {
                is IBGSdkCoreEvent.FeaturesFetched -> handleFeatureStateChange()
                else -> Unit
            }
        }
    }

    override fun contribute(
        report: State,
        spanSelector: SpanSelector<HubDirectory, HubLaunchDirectory>
    ): Future<Boolean> = submitOnStoreQueue {
        runCatching {
            "[Hub] $logsPrefix is being asked to contribute to report".logDWithThreadName()
            blockingRetrieve(JSONArrayDataAggregator(), spanSelector)
                ?.also { result -> "[Hub] Found ${result.length()} logs in $logsPrefix for report contribution".logVWithThreadName() }
                ?.toString()
                ?.let(report::setNetworkLogs)
                ?: run {
                    "[Hub] Contribution from $logsPrefix to report produced null, falling back to default.".logDWithThreadName()
                    JSONArray().toString().let(report::setNetworkLogs)
                }; true
        }.getOrLogAndReport(false, "[Hub] $logsPrefix store wasn't able to contribute to report.")
    }

    private fun handleFeatureStateChange() {
        runCatching {
            takeIf { isFeatureEnabled() }
                ?.also { "[Hub] ${logsPrefix}feature is enabled, initializing".logVWithThreadName() }
                ?.blockingInit()
                ?: run {
                    "[Hub] $logsPrefix feature is disabled, deleting ...".logVWithThreadName()
                    blockingDelete(batcher, ActiveCurrentSpanSelector())
                }
        }.runOrLogAndReport("[Hub] Error while handing $logsPrefix feature state changes.")
    }

    private fun isFeatureEnabled() = InstabugCore.isFeatureEnabled(IBGFeature.NETWORK_LOGS)

    companion object {
        fun createBatcher(limitsApplier: LimitConstraintApplier): LogsBatcher =
            SimpleLogsBatcher(NETWORK_LOGS_STORE_LIMIT, limitsApplier)
    }
}