package com.instabug.library.datahub

import androidx.annotation.VisibleForTesting
import com.instabug.library.internal.filestore.Directory
import com.instabug.library.internal.filestore.FileOperation
import com.instabug.library.util.LimitConstraintApplier
import com.instabug.library.util.TimeUtils
import kotlin.math.roundToInt

interface LogsBatcher {
    val currentBatchName: String
    fun getBatchingOps(): List<FileOperation<Directory, Unit>>
    fun onBatchingOpsExecuted()
    fun onLogStored()
    fun reset()
}

@VisibleForTesting
const val DEFAULT_BATCH_SIZE_PERCENTAGE = 0.25f

class SimpleLogsBatcher(
    private val totalLimit: Int,
    private val limitsApplier: LimitConstraintApplier
) : LogsBatcher {

    private val batchSize: Int
        get() = (limitsApplier.applyConstraints(totalLimit) * DEFAULT_BATCH_SIZE_PERCENTAGE).roundToInt()

    private val allowedNumberOfBatches: Int
        get() = (1 / DEFAULT_BATCH_SIZE_PERCENTAGE).toInt().inc()

    override var currentBatchName: String = TimeUtils.currentTimeMillis().toString()
        private set

    private var currentBatchEntriesCount: Int = 0

    private var batchesCount: Int = 0

    private var shouldDeleteBatchOnNext: Boolean = false
    private var shouldCreateNewBatchOnNext: Boolean = true

    override fun getBatchingOps(): List<FileOperation<Directory, Unit>> = buildList {
        DeleteOldestBatchFileOp
            .takeIf { shouldDeleteBatchOnNext }
            ?.also(this::add)
        takeIf { shouldCreateNewBatchOnNext }
            ?.let { CreateNewBatchFile(currentBatchName) }
            ?.also(this::add)
    }

    override fun onBatchingOpsExecuted() {
        takeIf { shouldDeleteBatchOnNext }
            ?.also { shouldDeleteBatchOnNext = false }
            ?.also { batchesCount-- }

        takeIf { shouldCreateNewBatchOnNext }
            ?.also { shouldCreateNewBatchOnNext = false }
            ?.also { batchesCount++ }
            ?.also { currentBatchEntriesCount = 0 }
    }

    override fun onLogStored() {
        currentBatchEntriesCount++
        shouldDeleteBatchOnNext = shouldDeleteBatch()
        shouldCreateNewBatchOnNext = shouldCreateNewBatch()
        if (shouldCreateNewBatchOnNext) currentBatchName = TimeUtils.currentTimeMillis().toString()
    }

    override fun reset() {
        shouldDeleteBatchOnNext = false
        shouldCreateNewBatchOnNext = true
        currentBatchEntriesCount = 0
        batchesCount = 0
        currentBatchName = TimeUtils.currentTimeMillis().toString()
    }

    private fun shouldDeleteBatch() = (batchesCount > allowedNumberOfBatches) ||
            (batchesCount == allowedNumberOfBatches && currentBatchEntriesCount >= batchSize)

    private fun shouldCreateNewBatch() = currentBatchEntriesCount >= batchSize
}
