package com.instabug.library.datahub

import android.content.Context
import androidx.annotation.WorkerThread
import com.instabug.library.internal.filestore.Directory
import com.instabug.library.internal.filestore.SpanDirectoriesOwner
import com.instabug.library.internal.filestore.SpansDirectory
import com.instabug.library.internal.filestore.SpansDirectoryFactory
import com.instabug.library.util.extenstions.getOrLogAndReport
import com.instabug.library.util.extenstions.toDirectory
import java.io.File

private const val DATA_HUB_DIR_NAME = "data-hub"

/**
 * Represents the "data-hub" directory along with its content. Also is a realization of [SpanDirectoriesOwner]
 * contract because "data-hub" directory is launch-span based.
 * @constructor takes a [File] representing the direct parent of the directory & a [String] representing
 * the current span id if any.
 */
class HubDirectory(
    parent: Directory,
    val currentLaunchId: String?
) : SpansDirectory<HubLaunchDirectory>(parent, DATA_HUB_DIR_NAME) {
    override val currentDirectory: HubLaunchDirectory?
        get() = currentLaunchId?.let { launchId -> HubLaunchDirectory(this, launchId) }
    override val oldDirectories: List<HubLaunchDirectory>
        @WorkerThread get() = runCatching {
            listFiles { file -> file.isDirectory && file.name != currentLaunchId }
                ?.map { file -> HubLaunchDirectory(this, file.name) }.orEmpty()
        }.getOrLogAndReport(emptyList(), "Error while getting hub old directories.")

    class Factory(
        private val ctxGetter: () -> Context?,
        private val rootDirGetter: (Context) -> File?
    ) : SpansDirectoryFactory<HubDirectory> {
        private var currentSessionId: String? = null

        override fun setCurrentSpanId(spanId: String?) {
            currentSessionId = spanId
        }

        override fun invoke(): HubDirectory? =
            ctxGetter()?.let(rootDirGetter)
                ?.toDirectory()
                ?.let { rootDir -> HubDirectory(rootDir, currentSessionId) }
    }
}

private const val NETWORK_LOGS_DIR_NAME = "network-logs"
private const val IBG_LOGS_DIR_NAME = "ibg-logs"

/**
 * Represents a launch directory of the "data-hub"
 * @constructor takes a [HubDirectory] as its direct parent & a [String] which is the launch id this
 * directory represents
 */
class HubLaunchDirectory(
    parent: HubDirectory,
    val launchId: String
) : Directory(parent, launchId) {
    val networkLogsDirectory: Directory
        get() = Directory(this, NETWORK_LOGS_DIR_NAME)

    val ibgLogsDirectory: Directory
        get() = Directory(this, IBG_LOGS_DIR_NAME)
}