package com.instabug.library.datahub

import androidx.annotation.WorkerThread
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent
import com.instabug.library.internal.filestore.ActiveCurrentSpanSelector
import com.instabug.library.internal.filestore.ActiveDirectorySelector
import com.instabug.library.internal.filestore.DataAggregator
import com.instabug.library.internal.filestore.DeleteDirectory
import com.instabug.library.internal.filestore.Directory
import com.instabug.library.internal.filestore.DirectorySelector
import com.instabug.library.internal.filestore.FileOperation
import com.instabug.library.internal.filestore.MakeDirectory
import com.instabug.library.internal.filestore.OldSpansSelector
import com.instabug.library.internal.filestore.OperationScopeBuilder
import com.instabug.library.internal.filestore.SpanSelector
import com.instabug.library.internal.filestore.SpansFileDataStore
import com.instabug.library.internal.filestore.SpansFileDataStoreLifecycle
import com.instabug.library.model.State
import com.instabug.library.util.extenstions.getOrLogAndReport
import com.instabug.library.util.extenstions.logDWithThreadName
import com.instabug.library.util.extenstions.logVWithThreadName
import com.instabug.library.util.extenstions.runOrLogAndReport
import com.instabug.library.util.threading.OrderedExecutorService
import java.util.concurrent.Future

// region Data Stores
interface HubDataStore<T : DataHubLog> : SpansFileDataStore<T, HubDirectory, HubLaunchDirectory>

interface HubControlledDataStore<T : DataHubLog> :
    HubDataStore<T>,
    SpansFileDataStoreLifecycle<HubDirectory>

abstract class HubBatchedDataStore<T : DataHubLog>(
    private val executor: OrderedExecutorService,
    protected val batcher: LogsBatcher
) : HubControlledDataStore<T> {
    protected abstract val storeDirectorySelector: DirectorySelector<HubLaunchDirectory, Directory>

    protected abstract val executionQueue: String

    protected abstract val logsPrefix: String

    private var operationsDirectory: HubDirectory? = null

    override fun init(operationsDirectory: HubDirectory): Future<Boolean> =
        submitOnStoreQueue {
            runCatching {
                "[Hub] $logsPrefix data store is being initialized with ops dir $operationsDirectory".logDWithThreadName()
                blockingInit(operationsDirectory); true
            }.getOrLogAndReport(false, "[Hub] Error while initializing $logsPrefix data store.")
        }

    @WorkerThread
    protected fun blockingInit(operationsDirectory: HubDirectory) {
        this.operationsDirectory = operationsDirectory
        blockingInit()
    }

    @WorkerThread
    protected open fun blockingInit() {
        OperationScopeBuilder(MakeDirectory())
            .inDirectory(storeDirectorySelector)
            .onSpan(ActiveCurrentSpanSelector())
            .buildAndExecute(operationsDirectory)
    }

    override fun onSpanStarted(spanId: String) {
        // Hub stores is following launch spans. Hence, it shouldn't be responsive to span starts.
        // Instead init method should be doing span start duties as part of its flow.
    }

    override fun store(log: T) {
        executeOnStoreQueue {
            runCatching {
                "[Hub] Inserting log in $logsPrefix data store".logDWithThreadName()
                operateOnStoreDirectory(
                    StoreOnBatchedDirectoryOp(log, batcher),
                    ActiveCurrentSpanSelector()
                )
            }.runOrLogAndReport("[Hub] Error while store log in $logsPrefix data store.")
        }
    }

    override fun <Out> retrieve(
        aggregator: DataAggregator<Out>,
        spanSelector: SpanSelector<HubDirectory, HubLaunchDirectory>
    ): Future<Out?> = submitOnStoreQueue {
        runCatching {
            "[Hub] Retrieving data from $logsPrefix data store".logDWithThreadName()
            blockingRetrieve(aggregator, spanSelector)
        }.getOrLogAndReport(null, "[Hub] Error while retrieving data from $logsPrefix data store.")
    }

    @WorkerThread
    protected fun <Out> blockingRetrieve(
        aggregator: DataAggregator<Out>,
        spanSelector: SpanSelector<HubDirectory, HubLaunchDirectory>
    ): Out? = operateOnStoreDirectory(ReadBatchedLogs(aggregator), spanSelector)

    override fun clear() {
        executeOnStoreQueue {
            runCatching {
                "[Hub] Clearing $logsPrefix data store".logDWithThreadName()
                operateOnStoreDirectory(
                    ClearBatchedDirectoryOp(batcher),
                    ActiveCurrentSpanSelector()
                )
            }.runOrLogAndReport("Error while clearing batched data store.")
        }
    }

    override fun delete() {
        executeOnStoreQueue {
            runCatching {
                "[Hub] Deleting $logsPrefix data store".logDWithThreadName()
                blockingDelete(batcher, ActiveCurrentSpanSelector())
            }.runOrLogAndReport("[Hub] Error while deleting $logsPrefix data store.")
        }
    }

    @WorkerThread
    protected fun blockingDelete(
        batcher: LogsBatcher?,
        spanSelector: SpanSelector<HubDirectory, HubLaunchDirectory>
    ): Unit? = operateOnStoreDirectory(DeleteBatchedDirectoryOp(batcher), spanSelector)

    override fun cleanse(): Future<Boolean> =
        submitOnStoreQueue {
            runCatching {
                "[Hub] Cleansing $logsPrefix data store".logDWithThreadName()
                blockingDeleteOld(); true
            }.getOrLogAndReport(false, "[Hub] Error while cleansing $logsPrefix data store.")
        }

    override fun onSpanEnded() {
        // Hub stores is following launch spans. Hence, it shouldn't be responsive to span ends.
    }

    override fun shutdown(): Future<Boolean> =
        submitOnStoreQueue {
            runCatching {
                "[Hub] Shutting down $logsPrefix data store".logDWithThreadName()
                blockingDelete(batcher, ActiveCurrentSpanSelector())
                blockingDeleteOld()
                operationsDirectory = null; true
            }.onFailure { operationsDirectory = null }
                .getOrLogAndReport(false, "[Hub] Error while shutting down $logsPrefix data store.")
        }

    @WorkerThread
    protected fun blockingDeleteOld() {
        OperationScopeBuilder(DeleteDirectory())
            .inDirectory(storeDirectorySelector)
            .onMultiSpans(OldSpansSelector())
            .buildAndExecute(operationsDirectory)
    }

    protected fun executeOnStoreQueue(operation: () -> Unit) {
        executor.execute(executionQueue, operation)
    }

    protected fun <Out> submitOnStoreQueue(operation: () -> Out) =
        executor.submit(executionQueue, operation)

    @WorkerThread
    protected fun <Out> operateOnStoreDirectory(
        operation: FileOperation<Directory, Out>,
        spanSelector: SpanSelector<HubDirectory, HubLaunchDirectory>
    ) = OperationScopeBuilder(operation)
        .inDirectory(ActiveDirectorySelector(storeDirectorySelector))
        .onSpan(spanSelector)
        .buildAndExecute(operationsDirectory)
        ?: run { "Operations directory is null (shutdown) or operation exec yielded null".logVWithThreadName(); null }
}

// endregion

// region Extra Contracts
interface HubEventsListener {
    fun onNewEvent(event: IBGSdkCoreEvent)
}

interface ReportContributor {
    fun contribute(
        report: State,
        spanSelector: SpanSelector<HubDirectory, HubLaunchDirectory>
    ): Future<Boolean>
}

//endregion