package com.instabug.library.util.threading;

import static android.os.Process.THREAD_PRIORITY_BACKGROUND;

import android.content.Context;
import android.os.Looper;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RestrictTo;

import com.instabug.library.Constants;
import com.instabug.library.Instabug;
import com.instabug.library.util.InstabugSDKLogger;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.FutureTask;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * Singleton class for thread pool provider
 */
@RestrictTo(RestrictTo.Scope.LIBRARY_GROUP)
public class PoolProvider {
    /*
     * Number of cores to decide the number of threads
     */
    public static final int NUMBER_OF_CORES = Runtime.getRuntime().availableProcessors();
    private static final int NUMBER_OF_THREADS_FOR_SCHEDULED_EXECUTOR = 4;


    /*
     * thread pool executor for IO background tasks
     */
    private final ThreadPoolExecutor forIOTasks;

    /*
     * thread pool executor for scheduling background tasks
     */
    private final ScheduledThreadPoolExecutor forScheduledTasks;
    /*
     * thread pool executor for main thread tasks
     */
    private final Executor mainThreadExecutor;
    private final OrderedExecutorService orderedExecutor;

    /*
     * an instance of PoolProvider
     */
    @SuppressWarnings("squid:S3077") // Suppresses SonarQube warnings on volatile fields
    private static volatile PoolProvider instance;


//    private static final MonitoredSingleThreadExecutor databaseExecutor = new MonitoredSingleThreadExecutor();

    private static final Map<String, SingleThreadPoolExecutor> singleThreadPoolExecutorhMap = new HashMap<>();
    private static final Map<String, ReturnableSingleThreadExecutor> returnableSingleThreadPoolExecutorhMap = new HashMap<>();
    private static final Map<String, NetworkingSingleThreadPoolExecutor> networkingSingleThreadExecutorMap = new HashMap<>();
    private static final Map<String, MonitoredSingleThreadExecutor> monitoredSingleThreadExecutorMap = new HashMap<>();
    private static final ReturnableSingleThreadExecutor apiExecutor = new ReturnableSingleThreadExecutor("API-executor");
    private static final SingleThreadPoolExecutor sessionExecutor = new SingleThreadPoolExecutor("v3-session");
    private static final SingleThreadPoolExecutor visualUserStepsProvider = new SingleThreadPoolExecutor("steps-executor");
    private static final ReturnableSingleThreadExecutor sharedPrefExecutor = new ReturnableSingleThreadExecutor("shared-pref-executor");
    private static final SingleThreadPoolExecutor featuresFlagsCheckerExecutor = new SingleThreadPoolExecutor("features-flags-checker-executor");

    private static final Object lock = new Object();

    /*
     * returns the instance of PoolProvider
     */
    public static PoolProvider getInstance() {
        PoolProvider localInstance1 = instance;
        if (localInstance1 != null) return localInstance1;
        synchronized (lock) {
            PoolProvider localInstance2 = instance;
            if (localInstance2 != null) return localInstance2;
            PoolProvider initializedInstance = new PoolProvider();
            instance = initializedInstance;
            return initializedInstance;
        }
    }

    @Nullable
    public static Context getContext() {
        try {
            return Instabug.getApplicationContext();
        } catch (IllegalStateException exception) {
            return null;
        }
    }

    /*
     * constructor for  PoolProvider
     */
    private PoolProvider() {
        // setting the thread pool executor for forIOTasks;
        int coreThreadsCount = Math.min(8, (NUMBER_OF_CORES * 2));
        forIOTasks = new ThreadPoolExecutor(
                coreThreadsCount,
                coreThreadsCount,
                10L,
                TimeUnit.SECONDS,
                new LinkedBlockingQueue<>(),
                new PriorityThreadFactory("core-io-executor", THREAD_PRIORITY_BACKGROUND)
        );

        // setting the thread pool executor for forScheduledTasks
        forScheduledTasks = new ScheduledThreadPoolExecutor(
                NUMBER_OF_THREADS_FOR_SCHEDULED_EXECUTOR,
                new PriorityThreadFactory("core-scheduled-executor", THREAD_PRIORITY_BACKGROUND));

        // setting the thread pool executor for mainThreadExecutor;
        mainThreadExecutor = new MainThreadExecutor();

        orderedExecutor = new BasicOrderedExecutorService(forIOTasks);
    }


    /*
     * Post a task for IO operations
     */
    public static void postIOTask(final Runnable runnable) {
        if (runnable != null) {
            getInstance().orderedExecutor.execute(() -> {
                if (getContext() == null) {
                    return;
                }
                try {
                    runnable.run();
                } catch (Throwable throwable) {
                    if (throwable instanceof OutOfMemoryError)
                        InstabugSDKLogger.e(Constants.LOG_TAG, "low memory, can't run i/o task", throwable);
                    else
                        InstabugSDKLogger.e(Constants.LOG_TAG, "Error while running IO task", throwable);

                }
            });
        }
    }

    /*
     * Post a task for IO operations
     */
    public static void postOrderedIOTask(@NonNull String key, final Runnable runnable) {
        if (runnable != null) {
            getInstance().orderedExecutor.execute(key, new Runnable() {
                @Override
                public void run() {
                    if (getContext() == null) {
                        return;
                    }
                    try {
                        runnable.run();
                    } catch (Throwable throwable) {
                        if (throwable instanceof OutOfMemoryError)
                            InstabugSDKLogger.e(Constants.LOG_TAG, "low memory, can't run i/o task", throwable);
                        else
                            InstabugSDKLogger.e(Constants.LOG_TAG, "Error while running IO task", throwable);

                    }
                }
            });
        }
    }

    /*
     * Post a task for IO operations only if the calling thread is the Main thread
     *  to avoid multiple useless thread calls
     */
    public static void postIOTaskWithCheck(final Runnable runnable) {
        if (runnable != null) {
            if (Looper.myLooper() == Looper.getMainLooper()) {
                postIOTask(runnable);
            } else {
                runnable.run();
            }
        }
    }

    public static <T> Future<T> submitIOTask(Callable<T> callable) {
        return getInstance().forIOTasks.submit(callable);
    }

    public static <T> Future<T> submitOrderedIOTask(String key, Callable<T> callable) {
        return getInstance().orderedExecutor.submit(key, callable);
    }


    /*
     * Post a task for delayed operations
     */
    public static void postDelayedTask(final Runnable runnable, long delayMillis) {
        if (runnable != null) {
            getInstance().forScheduledTasks.schedule(new Runnable() {
                @Override
                public void run() {
                    if (getContext() == null) {
                        return;
                    }
                    try {
                        runnable.run();
                    } catch (OutOfMemoryError outOfMemoryError) {
                        InstabugSDKLogger.e(Constants.LOG_TAG, "low memory, can't run delayed task", outOfMemoryError);
                    }
                }

            }, delayMillis, TimeUnit.MILLISECONDS);
        }
    }

    @NonNull
    public static ScheduledFuture<?> postDelayedTaskAtFixedDelay(
            long initialDelay,
            long delayMillis,
            @NonNull final Runnable command
    ) {
        return getInstance().forScheduledTasks.scheduleWithFixedDelay(() -> {
            if (getContext() == null) return;
            DefensiveRunnableKt.runDefensive(command).run();

        }, initialDelay, delayMillis, TimeUnit.MILLISECONDS);
    }

    /*
     * Post a task for Main thread operations without context check
     */
    public static void postMainThreadTaskWithoutCheck(final Runnable runnable) {
        if (runnable != null) {
            getInstance().mainThreadExecutor.execute(() -> {
                try {
                    runnable.run();
                } catch (OutOfMemoryError outOfMemoryError) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "low memory, can't run main thread task", outOfMemoryError);
                }

            });
        }
    }

    /*
     * Post a task for Main Thread operations
     */
    public static void postMainThreadTask(final Runnable runnable) {
        if (runnable != null) {
            getInstance().mainThreadExecutor.execute(() -> {
                if (getContext() == null) {
                    return;
                }
                try {
                    runnable.run();
                } catch (OutOfMemoryError outOfMemoryError) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "low memory, can't run main thread task", outOfMemoryError);
                }
            });
        }
    }

    /*
     * Submit a task on the Main thread
     */
    public static <R> FutureTask<R> submitMainThreadTask(Callable<R> task) {
        FutureTask<R> future = new FutureTask<>(task);
        postMainThreadTask(future);
        return future;
    }

    /*
     * Post a task using specific executor
     */
    public static void postTask(Executor executor, final Runnable runnable) {
        if (executor != null && runnable != null) {
            executor.execute(() -> {
                if (getContext() == null) {
                    return;
                }
                try {
                    runnable.run();
                } catch (OutOfMemoryError outOfMemoryError) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "low memory, can't run task", outOfMemoryError);
                }
            });
        }
    }


    public static synchronized Executor getSingleThreadExecutor(String identifier) {
        if (singleThreadPoolExecutorhMap.containsKey(identifier)) {
            return singleThreadPoolExecutorhMap.get(identifier);
        } else {
            final SingleThreadPoolExecutor singleThreadPoolExecutor = new SingleThreadPoolExecutor(identifier);
            singleThreadPoolExecutor.setIdentifier(identifier)
                    .setThreadPoolIdleListener(new ThreadPoolIdleListener() {
                        @Override
                        public void onPoolReachIdleState(@Nullable String identifier) {
                            if (identifier != null) {
                                singleThreadPoolExecutorhMap.remove(identifier);
                            }
                        }
                    });
            singleThreadPoolExecutorhMap.put(identifier, singleThreadPoolExecutor);
            return singleThreadPoolExecutor;
        }
    }

    public static synchronized ExecutorService getNetworkingSingleThreadExecutorService(String identifier) {
        return getNetworkingSingleThreadExecutor(identifier, false);
    }

    public static synchronized Executor getNetworkingSingleThreadExecutor(String identifier) {
        return getNetworkingSingleThreadExecutor(identifier, false);
    }

    public static synchronized ExecutorService getNetworkingSingleThreadExecutor(String identifier, boolean retryable) {
        if (networkingSingleThreadExecutorMap.containsKey(identifier)) {
            return networkingSingleThreadExecutorMap.get(identifier);
        } else {
            NetworkingSingleThreadPoolExecutor singleThreadPoolExecutor;
            if (retryable) {
                singleThreadPoolExecutor = new RetryableNetworkingSingleThreadExecutor(identifier);
            } else {
                singleThreadPoolExecutor = new NetworkingSingleThreadPoolExecutor(identifier);
            }

            singleThreadPoolExecutor.setIdentifier(identifier).setThreadPoolIdleListener(new ThreadPoolIdleListener() {
                @Override
                public void onPoolReachIdleState(@Nullable String identifier) {
                    if (identifier != null) {
                        singleThreadPoolExecutorhMap.remove(identifier);
                    }
                }
            });
            networkingSingleThreadExecutorMap.put(identifier, singleThreadPoolExecutor);
            return singleThreadPoolExecutor;
        }
    }

    public static synchronized ReturnableSingleThreadExecutor getReturnableSingleThreadExecutor(String identifier) {
        if (returnableSingleThreadPoolExecutorhMap.containsKey(identifier)) {
            return returnableSingleThreadPoolExecutorhMap.get(identifier);
        } else {
            final ReturnableSingleThreadExecutor singleThreadPoolExecutor = new ReturnableSingleThreadExecutor(identifier);
            returnableSingleThreadPoolExecutorhMap.put(identifier, singleThreadPoolExecutor);
            return singleThreadPoolExecutor;
        }
    }

    public static synchronized MonitoredSingleThreadExecutor getMonitoredSingleThreadExecutor(String identifier) {
        if (monitoredSingleThreadExecutorMap.containsKey(identifier)) {
            return monitoredSingleThreadExecutorMap.get(identifier);
        } else {
            final MonitoredSingleThreadExecutor singleThreadPoolExecutor = new MonitoredSingleThreadExecutor();
            monitoredSingleThreadExecutorMap.put(identifier, singleThreadPoolExecutor);
            return singleThreadPoolExecutor;
        }
    }

    public static Executor getUserActionsExecutor() {
        return getSingleThreadExecutor("user-actions-executor");
    }

    public ThreadPoolExecutor getBackgroundExecutor() {
        return forIOTasks;
    }

    public static synchronized Executor getSyncExecutor() {
        return getSingleThreadExecutor("sync-Executor");
    }

    public ThreadPoolExecutor getIOExecutor() {
        return forIOTasks;
    }

    public Executor getMainThreadExecutor() {
        return mainThreadExecutor;
    }

    public OrderedExecutorService getOrderedExecutor() {
        return orderedExecutor;
    }

    public ScheduledExecutorService getScheduledExecutor() {
        return forScheduledTasks;
    }

    public static ReturnableSingleThreadExecutor getApiExecutor() {
        return apiExecutor;
    }

    public static SingleThreadPoolExecutor getSessionExecutor() {
        return sessionExecutor;
    }

    public static SingleThreadPoolExecutor getVisualUserStepsProvider() {
        return visualUserStepsProvider;
    }

    public static ReturnableSingleThreadExecutor getSharedPrefExecutor() {
        return sharedPrefExecutor;
    }

    public static ReturnableSingleThreadExecutor getChatsCacheExecutor() {
        return getReturnableSingleThreadExecutor("chats-cache-executor");
    }

    public static ReturnableSingleThreadExecutor getSurveysDBExecutor() {
        return getReturnableSingleThreadExecutor("surveys-db-executor");
    }

    public static MonitoredSingleThreadExecutor getDatabaseExecutor() {
        return getMonitoredSingleThreadExecutor("IBG-db-executor");
    }

    public static MonitoredSingleThreadExecutor getDiagnosticsDatabaseExecutor() {
        return getMonitoredSingleThreadExecutor("IBG-diagnostics-db-executor");
    }

    public static ReturnableSingleThreadExecutor getFilesEncryptionExecutor() {
        return getReturnableSingleThreadExecutor("Files-Encryption");
    }

    public Executor getMainExecutor() {
        return mainThreadExecutor;
    }


    public static Executor getBugsExecutor() {
        return getSingleThreadExecutor("bugs-executor");
    }

    public static SingleThreadPoolExecutor getFeaturesFlagsCheckerExecutor() {
        return featuresFlagsCheckerExecutor;
    }
}
