package com.instabug.library.util.threading

import android.util.Log
import com.instabug.library.Constants
import com.instabug.library.diagnostics.IBGDiagnostics
import com.instabug.library.util.InstabugSDKLogger

/**
 *@author mhashim6 on 3/4/22
 */
inline fun runGracefully(
	logMsg: String = "Encountered error in running action",
	expecting: (Throwable) -> Unit = { Log.e(Constants.LOG_TAG, "$logMsg. cause: $it") },
	explosive: () -> Unit,
) {
	runCatching(explosive).onFailure(expecting)
}

fun nonFatal(oom: OutOfMemoryError) {
	runGracefully(logMsg = "Error Reporting OOM NonFatal", expecting = { defensiveLog(it) }) {
        IBGDiagnostics.reportNonFatal(
            oom,
            errorMsgOf(oom)
        )
    }
}

fun defensiveLog(
	error: Throwable,
	msg: String = "Error in Single Thread Executor(${Thread.currentThread().name})"
) {
	runCatching { InstabugSDKLogger.e(Constants.LOG_TAG, "$msg. cause: $error") }.onFailure {
		Log.e(Constants.LOG_TAG, "Failed to log Throwable: $error. Log msg: $msg")
	}

}

fun reportOOM(oom: OutOfMemoryError) {
	try {
		nonFatal(oom)
	} catch (beyondRedemption: Throwable) {
		defensiveLog(
			beyondRedemption,
			"Failed to report non-fatal in Single Thread Executor(${Thread.currentThread().name}), cause: $beyondRedemption"
		)
	}
}

inline fun <T> runDefensive(crossinline action: () -> T?) = Runnable {
	try {
		action.invoke()
	} catch (error: Throwable) {
		defensiveLog(error)
		error.takeIf { it is OutOfMemoryError }
			?.let { it as OutOfMemoryError }
			?.also { reportOOM(it) }
	}
}

fun runDefensive(runnable: Runnable?) = runDefensive {
	runnable?.run()
}

fun errorMsgOf(e: Throwable) = when (e) {
	is OutOfMemoryError -> "OOM in Single Thread Executor(${Thread.currentThread().name}). cause: $e"
	else -> "Error in Single Thread Executor(${Thread.currentThread().name}). cause: $e"
}
