package com.instabug.library.tracking;

import static com.instabug.library.model.StepType.ACTIVITY_CREATED;
import static com.instabug.library.model.StepType.ACTIVITY_DESTROYED;
import static com.instabug.library.model.StepType.ACTIVITY_PAUSED;
import static com.instabug.library.model.StepType.ACTIVITY_RESUMED;
import static com.instabug.library.model.StepType.ACTIVITY_STARTED;
import static com.instabug.library.model.StepType.ACTIVITY_STOPPED;
import static com.instabug.library.model.StepType.APPLICATION_CREATED;
import static com.instabug.library.model.StepType.FRAGMENT_ATTACHED;
import static com.instabug.library.model.StepType.FRAGMENT_DETACHED;
import static com.instabug.library.model.StepType.FRAGMENT_PAUSED;
import static com.instabug.library.model.StepType.FRAGMENT_RESUMED;
import static com.instabug.library.model.StepType.FRAGMENT_STARTED;
import static com.instabug.library.model.StepType.FRAGMENT_STOPPED;
import static com.instabug.library.model.StepType.FRAGMENT_VIEW_CREATED;
import static com.instabug.library.model.StepType.FRAGMENT_VISIBILITY_CHANGED;

import android.app.Activity;
import android.app.Application;
import android.content.res.Configuration;
import android.view.MotionEvent;
import android.view.Window;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.fragment.app.Fragment;

import com.instabug.library.Constants;
import com.instabug.library.Feature;
import com.instabug.library.IBGFeature;
import com.instabug.library.InstabugFeaturesManager;
import com.instabug.library.InstabugState;
import com.instabug.library.InstabugStateProvider;
import com.instabug.library.Platform;
import com.instabug.library._InstabugActivity;
import com.instabug.library.core.CurrentFragmentLifeCycleEventBus;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.eventbus.CurrentActivityConfigurationChange;
import com.instabug.library.core.eventbus.CurrentActivityLifeCycleEventBus;
import com.instabug.library.internal.servicelocator.CoreServiceLocator;
import com.instabug.library.sessionreplay.di.SessionReplayServiceLocator;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.InstabugSDKLogger;

import java.lang.ref.WeakReference;
import java.util.concurrent.Future;

import kotlin.Unit;

/**
 * @author mesbah
 */
public class InstabugInternalTrackingDelegate {
    private boolean isRegistered = false;
    private volatile static InstabugInternalTrackingDelegate INSTANCE;
    private final CurrentActivityMonitor currentActivityMonitor;
    private final InstabugActivityLifecycleListener activityLifecycleListener;
    @Nullable
    private WeakReference<Fragment> lastResumedFragment;

    @Nullable
    private volatile WeakReference<Activity> currentActivity;
    @Nullable
    private volatile WeakReference<Activity> currentRealActivity;

    @Nullable
    private volatile WeakReference<Activity> lastStoppedActivity;

    /**
     * Created this variable here, this is the same one in {@link SessionManager} because getting it from session manager may cause deadlocks while starting custom traces
     */
    private int startedActivitiesNumber = 0;


    private static final String FRAGMENT_NAV_HOST_NAME = "androidx.navigation.fragment.NavHostFragment";

    private final int platform;

    public static void init(@NonNull Application application) {
        if (INSTANCE == null) {
            INSTANCE = new InstabugInternalTrackingDelegate(application);
        }
    }

    /**
     * Returns the current singleton instance of this class.
     *
     * @return singleton instance of InstabugInternalTrackingDelegate.
     */
    public static InstabugInternalTrackingDelegate getInstance() {
        return INSTANCE;
    }

    private InstabugInternalTrackingDelegate(@NonNull Application application) {
        platform = SettingsManager.getInstance().getCurrentPlatform();
        currentActivityMonitor = new CurrentActivityMonitor();
        activityLifecycleListener = new InstabugActivityLifecycleListener();
        currentActivityMonitor.startMonitoring(application);
        registerLifecycleListeners(application);
    }

    public void onApplicationCreated(Application application) {
        if (isUserTrackingStepsEnable()) {
            InstabugSDKLogger.v(Constants.LOG_TAG, application.getClass().getSimpleName() + " created");
            InstabugTrackingStepsProvider.getInstance()
                    .addActivityLifecycleStep(application.getClass().getName(), APPLICATION_CREATED);
        }
    }

    public int getStartedActivitiesNumber() {
        return startedActivitiesNumber;
    }

    void handleActivityCreatedEvent(Activity activity) {

        if (isNotInstabugActivity(activity)) {
            if (isUserTrackingStepsEnable()) {
                InstabugSDKLogger.v(Constants.LOG_TAG, activity.getClass().getSimpleName() + " created");
                InstabugTrackingStepsProvider.getInstance().addActivityLifecycleStep(activity.getClass().getName(), ACTIVITY_CREATED);
            }

            if (isInstabugInEnabledState() && platform == Platform.ANDROID) {
                CoreServiceLocator.getReproStepsProxy().addVisualUserStep(
                        ACTIVITY_CREATED,
                        activity.getClass().getSimpleName(),
                        activity.getClass().getName(), null
                );
            }
            // send current_activity_created broadcast.
            CurrentActivityLifeCycleEventBus.INSTANCE.post(ActivityLifeCycleEvent.CREATED);
        }
    }

    void handleActivityStartedEvent(Activity activity) {
        startedActivitiesNumber++;
        if (isNotInstabugActivity(activity)) {
            if (isUserTrackingStepsEnable()) {
                InstabugSDKLogger.v(Constants.LOG_TAG, activity.getClass().getSimpleName() + " started");
                InstabugTrackingStepsProvider.getInstance().addActivityLifecycleStep(activity.getClass().getName(), ACTIVITY_STARTED);
            }

            if (isInstabugInEnabledState() && platform == Platform.ANDROID) {
                CoreServiceLocator.getReproStepsProxy().addVisualUserStep(
                        ACTIVITY_STARTED,
                        activity.getClass().getSimpleName(),
                        activity.getClass().getName(), null
                );
            }
        }

        // send current_activity_started broadcast.
        CurrentActivityLifeCycleEventBus.INSTANCE.post(ActivityLifeCycleEvent.STARTED);

    }

    void handleActivityResumedEvent(Activity activity) {
        if (isNotInstabugActivity(activity)) {
            if (isUserTrackingStepsEnable()) {
                InstabugSDKLogger.v(Constants.LOG_TAG, activity.getClass().getSimpleName() + " resumed");
                InstabugTrackingStepsProvider.getInstance().addActivityLifecycleStep(activity.getClass().getName(), ACTIVITY_RESUMED);
            }
            if (isInstabugInEnabledState()) {
                CoreServiceLocator.getReproStepsProxy().addVisualUserStep(
                        ACTIVITY_RESUMED,
                        activity.getClass().getSimpleName(),
                        activity.getClass().getName(), null
                );
            }
            CoreServiceLocator.getNavigableViewsTracker().onActivityResumed(activity);
            // send current_activity_resumed broadcast.
            CurrentActivityLifeCycleEventBus.INSTANCE.post(ActivityLifeCycleEvent.RESUMED);

            IBGWindowCallbacksHandler.registerWindowCallbacksIfNeeded(activity);
            CurrentViewProvider.getInstance().setCurrentActivityView(activity.getClass().getName());
        }
    }

    public void setCurrentActivity(Activity activity) {
        currentRealActivity = new WeakReference<>(activity);
        if (isNotInstabugActivity(activity)) {
            this.currentActivity = new WeakReference<>(activity);
        }
    }

    void handleActivityPausedEvent(Activity activity) {
        Activity activityInstance = getCurrentActivity();
        if (isNotInstabugActivity(activity)) {
            if (activityInstance == null) {
                InstabugSDKLogger.w(Constants.LOG_TAG, "No activity was set earlier than this call. Doing nothing");
                return;
            }
            if (!activity.equals(activityInstance)) {
                InstabugSDKLogger.w(Constants.LOG_TAG, "You're trying to pause an activity that is not the current" +
                        " activity! Please make sure you're calling onCurrentActivityPaused and onCurrentActivityResumed on every " +
                        "activity");
                return;
            }
            String activityName = activity.getClass().getName();
            String screenName = activity.getClass().getSimpleName();
            if (isUserTrackingStepsEnable()) {
                InstabugSDKLogger.v(Constants.LOG_TAG, screenName + " paused");
                InstabugTrackingStepsProvider.getInstance().addActivityLifecycleStep(activityName, ACTIVITY_PAUSED);
            }
            if (isInstabugInEnabledState()) {
                CoreServiceLocator.getReproStepsProxy().addVisualUserStep(
                        ACTIVITY_PAUSED,
                        screenName,
                        activityName, null
                );
            }
            // send current_activity_paused broadcast.
            CurrentActivityLifeCycleEventBus.INSTANCE.post(ActivityLifeCycleEvent.PAUSED);
        }
        CoreServiceLocator.getNavigableViewsTracker().onActivityPaused(activity);
    }

    void handleActivityStoppedEvent(Activity activity) {
        startedActivitiesNumber--;
        lastStoppedActivity = new WeakReference<>(activity);
        if (isNotInstabugActivity(activity)) {
            if (isUserTrackingStepsEnable()) {
                InstabugSDKLogger.v(Constants.LOG_TAG, activity.getClass().getSimpleName() + " stopped");
                Future<Unit> stepFuture = InstabugTrackingStepsProvider.getInstance().addActivityLifecycleStep(activity.getClass().getName(), ACTIVITY_STOPPED);
                if (startedActivitiesNumber == 0) {
                    SessionReplayServiceLocator.getSessionReplayDelegate().setPendingLog(stepFuture);
                }
            }
            if (isInstabugInEnabledState()) {
                CoreServiceLocator.getReproStepsProxy().addVisualUserStep(ACTIVITY_STOPPED,
                        activity.getClass().getSimpleName(),
                        activity.getClass().getName(), null);
            }
        }

        // send current_activity_stopped broadcast.
        CurrentActivityLifeCycleEventBus.INSTANCE.post(ActivityLifeCycleEvent.STOPPED);
    }

    void handleActivityDestroyedEvent(Activity activity) {
        if (isNotInstabugActivity(activity)) {
            if (isUserTrackingStepsEnable()) {
                InstabugSDKLogger.v(Constants.LOG_TAG, activity.getClass().getSimpleName() + " destroyed");
                InstabugTrackingStepsProvider.getInstance().addActivityLifecycleStep(activity.getClass().getName(), ACTIVITY_DESTROYED);
            }
            if (isInstabugInEnabledState()) {
                CoreServiceLocator.getReproStepsProxy().addVisualUserStep(ACTIVITY_DESTROYED,
                        activity.getClass().getSimpleName(),
                        activity.getClass().getName(), null);
            }
            Activity currentActivity = getCurrentActivity();
            if (currentActivity != null && currentActivity == activity) {
                clearCurrentActivity();
            }
            // send current_activity_destroyed broadcast.
            CurrentActivityLifeCycleEventBus.INSTANCE.post(ActivityLifeCycleEvent.DESTROYED);
        }
    }

    void onFragmentAttached(androidx.fragment.app.Fragment fragment) {
        if (fragment == null || isNavHostFragment(fragment)) {
            return;
        }
        Activity activity = getCurrentActivity();
        if (activity != null) {
            if (isUserTrackingStepsEnable())
                InstabugTrackingStepsProvider.getInstance().addFragmentLifecycleStep(fragment.getClass().getName(),
                        activity.getClass().getName(), FRAGMENT_ATTACHED);
        }
        CurrentFragmentLifeCycleEventBus.getInstance().post(FragmentLifeCycleEvent.ATTACHED);
    }

    void onFragmentViewCreated(Fragment fragment) {
        if (fragment == null || isNavHostFragment(fragment)) {
            return;
        }
        Activity activity = getCurrentActivity();
        if (activity != null) {
            if (isUserTrackingStepsEnable())
                InstabugTrackingStepsProvider.getInstance().addFragmentLifecycleStep(fragment.getClass().getName(),
                        activity.getClass().getName(), FRAGMENT_VIEW_CREATED);
        }
        CurrentFragmentLifeCycleEventBus.getInstance().post(FragmentLifeCycleEvent.VIEW_CREATED);
    }

    void onFragmentStarted(androidx.fragment.app.Fragment fragment) {
        if (fragment == null || isNavHostFragment(fragment)) {
            return;
        }
        Activity activity = getCurrentActivity();
        if (activity != null) {
            if (isUserTrackingStepsEnable())
                InstabugTrackingStepsProvider.getInstance().addFragmentLifecycleStep(fragment.getClass().getName(),
                        activity.getClass().getName(), FRAGMENT_STARTED);
        }
        CurrentFragmentLifeCycleEventBus.getInstance().post(FragmentLifeCycleEvent.STARTED);
    }

    void onFragmentResumed(androidx.fragment.app.Fragment fragment) {
        if (fragment == null || isNavHostFragment(fragment)) {
            return;
        }
        lastResumedFragment = new WeakReference<>(fragment);
        Activity activity = getCurrentActivity();
        if (activity != null) {
            if (isUserTrackingStepsEnable()) {
                InstabugTrackingStepsProvider.getInstance().addFragmentLifecycleStep(fragment.getClass().getName(),
                        activity.getClass().getName(), FRAGMENT_RESUMED);
            }
        }

        // sometimes the Toolbar register its own callback and overrides our callback,
        // so we need to check if our callback was unregistered for any reason and re-register it.
        if (fragment.getActivity() != null) {
            IBGWindowCallbacksHandler.registerWindowCallbacksIfNeeded(fragment.getActivity());
        }
        CurrentFragmentLifeCycleEventBus.getInstance().post(FragmentLifeCycleEvent.RESUMED);
        CurrentViewProvider.getInstance().setCurrentFragmentView(fragment.getClass().getName());

    }


    void onFragmentPaused(androidx.fragment.app.Fragment fragment) {
        if (fragment == null || isNavHostFragment(fragment)) {
            return;
        }
        lastResumedFragment = null;
        Activity activity = getCurrentActivity();
        if (activity != null) {
            if (isUserTrackingStepsEnable()) {
                InstabugTrackingStepsProvider.getInstance().addFragmentLifecycleStep(fragment.getClass().getName(),
                        activity.getClass().getName(), FRAGMENT_PAUSED);
            }
        }
        CurrentFragmentLifeCycleEventBus.getInstance().post(FragmentLifeCycleEvent.PAUSED);
    }

    void onFragmentStopped(androidx.fragment.app.Fragment fragment) {
        if (fragment == null || isNavHostFragment(fragment)) {
            return;
        }
        Activity activity = getCurrentActivity();
        if (activity != null) {
            if (isUserTrackingStepsEnable())
                InstabugTrackingStepsProvider.getInstance().addFragmentLifecycleStep(fragment.getClass().getName(),
                        activity.getClass().getName(), FRAGMENT_STOPPED);
        }
        CurrentFragmentLifeCycleEventBus.getInstance().post(FragmentLifeCycleEvent.STOPPED);
    }

    void onFragmentDetached(androidx.fragment.app.Fragment fragment) {
        if (fragment == null || isNavHostFragment(fragment)) {
            return;
        }
        Activity activity = getCurrentActivity();
        if (activity != null) {
            if (isUserTrackingStepsEnable())
                InstabugTrackingStepsProvider.getInstance().addFragmentLifecycleStep(fragment.getClass().getName(),
                        activity.getClass().getName(), FRAGMENT_DETACHED);
        }
        CurrentFragmentLifeCycleEventBus.getInstance().post(FragmentLifeCycleEvent.DETACHED);

    }

    public void onFragmentVisibilityChanged(boolean isVisible, androidx.fragment.app.Fragment fragment) {
        if (fragment == null || isNavHostFragment(fragment)) {
            return;
        }
        Activity activity = getCurrentActivity();
        if (activity != null) {
            if (isUserTrackingStepsEnable())
                InstabugTrackingStepsProvider.getInstance().addFragmentLifecycleStep(fragment.getClass().getName(),
                        activity.getClass().getName(), "Fragment visibility: " + isVisible,
                        FRAGMENT_VISIBILITY_CHANGED);
        }
        if (isInstabugInEnabledState()) {
            CoreServiceLocator.getReproStepsProxy().addVisualUserStep(
                    FRAGMENT_VISIBILITY_CHANGED,
                    fragment.getClass().getSimpleName(),
                    fragment.getClass().getName(), null);
        }
    }

    private boolean isNavHostFragment(Fragment fragment) {
        if (fragment == null) {
            return false;
        }
        return FRAGMENT_NAV_HOST_NAME.equals(fragment.getClass().getName());
    }

    void trackTouchEvent(MotionEvent event) {
        InstabugTouchEventsTracker.getInstance().trackTouchEvent(event);
    }

    @Nullable
    public Activity getCurrentActivity() {
        try {
            return currentActivity == null ? null : currentActivity.get();
        } catch (Throwable t) {
            InstabugCore.reportError(t, "Error while retrieving current activity");
        }
        return null;
    }


    @Nullable
    public Activity getLastStoppedActivity() {
        try {
            WeakReference<Activity> stoppedActivity = lastStoppedActivity;
            return stoppedActivity == null ? null : stoppedActivity.get();
        } catch (Throwable t) {
            InstabugCore.reportError(t, "Error while retrieving stopped activity");
        }
        return null;
    }

    public void clearCurrentActivity() {
        try {
            if (currentActivity == null) return;
            currentActivity.clear();
        } catch (Throwable t) {
            InstabugCore.reportError(t, "Error while clearing current activity");
        }
    }

    @Nullable
    public Activity getCurrentRealActivity() {
        try {
            return currentRealActivity == null ? null : currentRealActivity.get();
        } catch (Throwable t) {
            InstabugCore.reportError(t, "Error while retrieving current real activity");
        }
        return null;
    }

    @Nullable
    public Activity getTargetActivity() {
        Activity target = null;
        Activity activityInstance = getCurrentActivity();
        if (activityInstance != null && activityInstance.getParent() != null) {
            target = activityInstance.getParent();
            while (target.getParent() != null) {
                target = target.getParent();
            }
        } else if (activityInstance != null) {
            target = activityInstance;
        }

        return target;
    }

    @Nullable
    public Object getLastSeenView() {
        if (lastResumedFragment != null && lastResumedFragment.get() != null) {
            return lastResumedFragment.get();
        } else {
            return getTargetActivity();
        }
    }

    public void registerLifecycleListeners(Application application) {
        application.registerActivityLifecycleCallbacks(activityLifecycleListener);
        application.registerComponentCallbacks(activityLifecycleListener);
        isRegistered = true;
    }


    public void unregisterLifecycleListeners(Application application) {
        InstabugSDKLogger.d(Constants.LOG_TAG, "Unregistering activity lifecycle listener");
        application.unregisterActivityLifecycleCallbacks(activityLifecycleListener);
        application.unregisterComponentCallbacks(activityLifecycleListener);
        isRegistered = false;
    }

    public boolean isRegistered() {
        return isRegistered;
    }

    private boolean isUserTrackingStepsEnable() {
        return InstabugFeaturesManager.getInstance().getFeatureState(IBGFeature.TRACK_USER_STEPS) == Feature.State.ENABLED
                && !InstabugStateProvider.getInstance().getState().equals(InstabugState.DISABLED);
    }

    private boolean isInstabugInEnabledState() {
        return InstabugStateProvider.getInstance().getState().equals(InstabugState.ENABLED);
    }

    private boolean isNotInstabugActivity(Activity activity) {
        return !(activity instanceof _InstabugActivity);
    }

    void handleConfigurationChanged(Configuration newConfig) {
        Activity targetActivity = getTargetActivity();
        if (targetActivity == null)
            return;
        // send current_activity_stopped broadcast.
        CurrentActivityConfigurationChange instance = CurrentActivityConfigurationChange.getInstance();
        instance.setNewConfig(newConfig);
        CurrentActivityConfigurationChange.getInstance().post(instance);
    }

}
