package com.instabug.library.sessionreplay.di

import com.instabug.library.Instabug
import com.instabug.library.Platform
import com.instabug.library.SessionSyncListener
import com.instabug.library.core.eventbus.V3SessionSyncResultEventBus
import com.instabug.library.internal.servicelocator.CoreServiceLocator
import com.instabug.library.internal.storage.AttachmentManager
import com.instabug.library.internal.storage.DiskUtils
import com.instabug.library.internal.storage.cache.dbv2.IBGDbManager
import com.instabug.library.logscollection.LogReceiver
import com.instabug.library.model.UserStep
import com.instabug.library.networkv2.INetworkManager
import com.instabug.library.networkv2.NetworkManager
import com.instabug.library.sessionV3.di.IBGSessionServiceLocator
import com.instabug.library.sessionV3.providers.FeatureSessionDataControllerHost
import com.instabug.library.sessionreplay.BasicSRStateChangeListenersRegistry
import com.instabug.library.sessionreplay.CompressSRDirsOperation
import com.instabug.library.sessionreplay.ControlledSRLogStore
import com.instabug.library.sessionreplay.ControlledSRScreenshotStore
import com.instabug.library.sessionreplay.DefaultSRLoggingController
import com.instabug.library.sessionreplay.DefaultSRMetadataDBHandler
import com.instabug.library.sessionreplay.DeleteSessionDirectoryOperation
import com.instabug.library.sessionreplay.RNSessionReplayUserEvaluatorImpl
import com.instabug.library.sessionreplay.SRDelegate
import com.instabug.library.sessionreplay.SRDelegateDependencies
import com.instabug.library.sessionreplay.SRDelegateImpl
import com.instabug.library.sessionreplay.SRDelegateWrapper
import com.instabug.library.sessionreplay.SRFilesDirectory
import com.instabug.library.sessionreplay.SRLoggingController
import com.instabug.library.sessionreplay.SRMetadataDBHandler
import com.instabug.library.sessionreplay.SRScreenshotStore
import com.instabug.library.sessionreplay.SRStateChangeListener
import com.instabug.library.sessionreplay.SRStateChangeListenersRegistry
import com.instabug.library.sessionreplay.SRSyncJob
import com.instabug.library.sessionreplay.SessionReplayRTConfigurationHandler
import com.instabug.library.sessionreplay.SessionReplayUserEvaluator
import com.instabug.library.sessionreplay.SessionReplayUserEvaluatorImpl
import com.instabug.library.sessionreplay.SessionSyncListenerWrapper
import com.instabug.library.sessionreplay.bitmap.SRBitMapCompressor
import com.instabug.library.sessionreplay.bitmap.SRBitmapScaler
import com.instabug.library.sessionreplay.configurations.SRConfigurationsHandler
import com.instabug.library.sessionreplay.configurations.SRLimitationsProvider
import com.instabug.library.sessionreplay.monitoring.AnalyticsSessionDataMapper
import com.instabug.library.sessionreplay.monitoring.ControlledSRLoggingMonitor
import com.instabug.library.sessionreplay.monitoring.MonitoringGarbageCollector
import com.instabug.library.sessionreplay.monitoring.QueuedSRMonitoringDelegate
import com.instabug.library.sessionreplay.monitoring.SRMonitoringDelegate
import com.instabug.library.sessionreplay.monitoring.SRMonitoringDirectory
import com.instabug.library.sessionreplay.monitoring.SRMonitoringSpansDataStore
import com.instabug.library.sessionreplay.monitoring.SRSessionDataController
import com.instabug.library.sessionreplay.monitoring.SRSessionDataControllerHost
import com.instabug.library.sessionreplay.monitoring.SimpleControlledSRLoggingMonitor
import com.instabug.library.sessionreplay.monitoring.ThrottledSRMonitoringSpansDataStore
import com.instabug.library.settings.SettingsManager
import com.instabug.library.tracking.CoreUserStepHandler
import com.instabug.library.tracking.UserStepsProvider
import com.instabug.library.util.LatestJobThrottler
import com.instabug.library.util.threading.OrderedExecutorService
import com.instabug.library.util.threading.PoolProvider
import com.instabug.library.visualusersteps.ReproCapturingProxy


object SessionReplayServiceLocator {
    private val srDependencies
        get() = object : SRDelegateDependencies {
            override val compressOperation = CompressSRDirsOperation()
            override val syncJob = SRSyncJob(
                networkManager = networkManager,
                metadataHandler = srMetadataDBHandler,
                filesDirectory = sessionReplayDirectory,
                configurations = srConfigurationsProvider
            )
            override val srDir = sessionReplayDirectory
            override val loggingController = srLoggingController
            override val executor = CoreServiceLocator.orderedExecutor
            override val metadataDBHandler = srMetadataDBHandler
            override val configurationsProvider = srConfigurationsProvider
            override val reproProxy: ReproCapturingProxy
                get() = CoreServiceLocator.reproCompositeProxy
            override val monitorDelegate: SRMonitoringDelegate
                get() = this@SessionReplayServiceLocator.monitorDelegate
            override val stateListener: SRStateChangeListener = stateListenersRegistry
            override val v3SessionSyncResultEventBus: V3SessionSyncResultEventBus
                get() = IBGSessionServiceLocator.v3SessionSyncResultEventBus
            override val srUserEvaluator: SessionReplayUserEvaluator
                get() = sessionReplayUserEvaluator
        }
    private val srDelegateWrapper by lazy {
        SRDelegateWrapper(
            featuresFlagsCheckerExecutor = PoolProvider.getFeaturesFlagsCheckerExecutor(),
            srDelegateFactory = { SRDelegateImpl(dependencies = srDependencies) },
            srConfigurationsHandler = srConfigurationsProvider
        )
    }

    @JvmStatic
    val sessionReplayDelegate: SRDelegate
        get() = srDelegateWrapper

    @JvmStatic
    val sessionReplayRTConfigurationHandler: SessionReplayRTConfigurationHandler
        get() = srDelegateWrapper
    val sessionReplayLogStore: ControlledSRLogStore by lazy {
        ControlledSRLogStore(
            executor = CoreServiceLocator.orderedExecutor,
            filesDirectory = this.sessionReplayDirectory,
            loggingController = srLoggingController,
            loggingMonitor = loggingMonitor
        )
    }

    val sessionReplayDirectory: SRFilesDirectory by lazy {
        SRFilesDirectory(
            CoreServiceLocator.orderedExecutor,
            Instabug::getApplicationContext,
            DiskUtils::getInstabugInternalDirectory
        )
    }
    val srLoggingController: SRLoggingController by lazy {
        DefaultSRLoggingController(this.srConfigurationsProvider)
    }

    @JvmStatic
    val monitorDelegate: SRMonitoringDelegate by lazy {
        val garbageCollector = MonitoringGarbageCollector(
            sessionCacheManager = IBGSessionServiceLocator.sessionCacheManger,
            dataStore = monitoringDataStore
        )
        QueuedSRMonitoringDelegate(
            CoreServiceLocator.orderedExecutor,
            loggingMonitor,
            monitoringDataStore,
            getMonitoringDirectoryFactory(),
            CoreServiceLocator.screenshotsAnalyticsEventBus,
            srConfigurationsProvider,
            garbageCollector,

            )
    }
    val srMetadataDBHandler: SRMetadataDBHandler by lazy {
        val dbManager = try {
            IBGDbManager.getInstance()
        } catch (e: Exception) {
            null
        }
        DefaultSRMetadataDBHandler(
            dbManager
        )
    }
    val srConfigurationsProvider: SRConfigurationsHandler by lazy {
        SRConfigurationsHandler.create()
    }

    val networkManager: INetworkManager by lazy { NetworkManager() }

    private val coreUserStepHandler by lazy { CoreUserStepHandler() }

    @JvmStatic
    val userStepsProvider: UserStepsProvider
        get() = coreUserStepHandler

    @JvmStatic
    val coreUserStepReceiver: LogReceiver<UserStep>
        get() = coreUserStepHandler

    @JvmStatic
    val limitationsProvider: SRLimitationsProvider
        get() = this.srConfigurationsProvider

    @JvmStatic
    val sessionReplayScreenshotStore: SRScreenshotStore
            by lazy {
                ControlledSRScreenshotStore(
                    sessionReplayLogStore,
                    SRBitmapScaler(),
                    sessionReplayDirectory,
                    SRBitMapCompressor(srConfigurationsProvider),
                    CoreServiceLocator.orderedExecutor,
                    srLoggingController,
                    loggingMonitor
                )
            }

    private val orderedExecutor: OrderedExecutorService
        get() = CoreServiceLocator.orderedExecutor

    val monitoringDataStore: SRMonitoringSpansDataStore by lazy {
        ThrottledSRMonitoringSpansDataStore(
            orderedExecutor,
            LatestJobThrottler(PoolProvider.getInstance().scheduledExecutor),
            getMonitoringDirectoryFactory()
        )
    }

    val loggingMonitor: ControlledSRLoggingMonitor by lazy {
        SimpleControlledSRLoggingMonitor(monitoringDataStore, srConfigurationsProvider)
    }

    val sessionDataControllerHost: FeatureSessionDataControllerHost by lazy {
        SRSessionDataController(
            dataStore = monitoringDataStore,
            dataMapper = AnalyticsSessionDataMapper()
        ).let(::SRSessionDataControllerHost)
    }

    private fun getMonitoringDirectoryFactory() =
        SRMonitoringDirectory.Factory(
            Instabug::getApplicationContext,
            AttachmentManager::getAttachmentInternalDirectory
        )

    val stateListenersRegistry: SRStateChangeListenersRegistry
            by lazy { BasicSRStateChangeListenersRegistry() }

    @JvmStatic
    var userSessionSyncListener: SessionSyncListener? = null
    val sessionSyncListener: SessionSyncListener by lazy {
        SessionSyncListenerWrapper(
            PoolProvider.getInstance().ioExecutor,
            srConfigurationsProvider
        ) { userSessionSyncListener }
    }

    val sessionReplayUserEvaluator: SessionReplayUserEvaluator by lazy {
        if (SettingsManager.getInstance().currentPlatform == Platform.RN) {
            rnSessionReplayUserEvaluator
        } else {
            nativeSessionReplayUserEvaluator
        }
    }

    private val nativeSessionReplayUserEvaluator
        get() = SessionReplayUserEvaluatorImpl(
            IBGSessionServiceLocator.sessionCacheManger,
            sessionSyncListener,
            srMetadataDBHandler,
            sessionReplayDirectory,
            ::DeleteSessionDirectoryOperation,
            monitoringDataStore,
            IBGSessionServiceLocator.syncJob,
        )

    private val rnSessionReplayUserEvaluator: SessionReplayUserEvaluator
        get() = RNSessionReplayUserEvaluatorImpl(
            nativeSessionReplayUserEvaluator,
            srConfigurationsProvider,
            PoolProvider.getInstance()
        )

}