package com.instabug.library.sessionreplay

import com.instabug.library.OnSessionReplayLinkReady
import com.instabug.library.core.InstabugCore
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent
import com.instabug.library.sessionreplay.configurations.SRConfigurationsHandler
import com.instabug.library.sessionreplay.configurations.SRConfigurationsProvider
import com.instabug.library.util.extenstions.logVerbose
import java.util.concurrent.Executor
import java.util.concurrent.Future

class SRDelegateWrapper(
    private val featuresFlagsCheckerExecutor: Executor,
    private val srDelegateFactory: () -> SRDelegate,
    private val srConfigurationsHandler: SRConfigurationsHandler,
) : SRDelegate, SessionReplayRTConfigurationHandler {
    private var srDelegate: SRDelegate? = null
    override fun setPendingLog(future: Future<Unit>) = featuresFlagsCheckerExecutor.execute {
        srDelegate
            .createIfNeeded()
            ?.setPendingLog(future)

    }

    private fun SRDelegate?.createIfNeeded(): SRDelegate? = this ?: srConfigurationsHandler
        .takeIf { isSessionReplayEnabled() }
        ?.let { srDelegateFactory() }
        .also { srDelegate = it }

    private fun isSessionReplayEnabled() =
        srConfigurationsHandler.srEnabled && InstabugCore.isV3SessionEnabled()

    override fun handleRuntimeConfigurationsChange(change: (SRConfigurationsProvider) -> Unit) {
        featuresFlagsCheckerExecutor.execute {
            change(srConfigurationsHandler)
            srDelegate
                .createIfNeeded()
                ?.handleConfigurationsChange()
            onConfigurationsHandled()
        }
    }

    override fun getRunningSessionLink(onSessionReplayLinkReady: OnSessionReplayLinkReady) {
        featuresFlagsCheckerExecutor.execute {
            srDelegate
                ?.getRunningSessionLink(onSessionReplayLinkReady)
                ?: onSessionReplayLinkReady.onSessionReplayLinkReady(null)
        }
    }

    override fun onNewEvent(event: IBGSdkCoreEvent) {
        featuresFlagsCheckerExecutor.execute {
            if (event is IBGSdkCoreEvent.FeaturesFetched) {
                handleFeaturesFetched(event)
            } else {
                srDelegate.createIfNeeded()?.onNewEvent(event)
            }
        }
    }

    private fun handleFeaturesFetched(event: IBGSdkCoreEvent.FeaturesFetched) {
        "Features configurations fetched, processing new configurations".logVerbose(tag = SR_LOG_TAG)
        srConfigurationsHandler.handleConfigurationsChange(event.response)
        srDelegate.createIfNeeded()
            ?.also { it.onNewEvent(event) }
            ?.handleConfigurationsChange()
        onConfigurationsHandled()
    }

    private fun onConfigurationsHandled() {
        if (!isSessionReplayEnabled()) {
            srDelegate = null
        }
    }
}