package com.instabug.library.model;


import android.content.Context;
import android.net.Uri;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.Constants;
import com.instabug.library.Feature;
import com.instabug.library.IBGFeature;
import com.instabug.library.Instabug;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.internal.storage.AttachmentsUtility;
import com.instabug.library.logging.InstabugLog;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.StringUtility;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * Created by tarek on 1/22/18.
 */

public class Report {


    private static final int ITEMS_LIST_LIMIT = 10;
    private static final int FILES_LIST_LIMIT = 3;

    private ArrayList<String> tags;
    private ArrayList<ConsoleLog> consoleLogs;

    public ArrayList<InstabugLog.LogMessage> getInstabugLogs() {
        return instabugLogs;
    }

    public void setInstabugLogs(ArrayList<InstabugLog.LogMessage> instabugLogs) {
        this.instabugLogs = instabugLogs;
    }

    private ArrayList<InstabugLog.LogMessage> instabugLogs;
    @Nullable
    private String userData;
    private HashMap<String, String> userAttributes;
    private HashMap<Uri, String> fileAttachments;

    public Report() {
        this.tags = new ArrayList<>();
        this.consoleLogs = new ArrayList<>();
        this.instabugLogs = new ArrayList<>();
        this.userAttributes = new HashMap<>();
        this.fileAttachments = new HashMap<>();
    }

    public void addTag(String... tag) {
        if (tag == null) return;
        int index = 0;
        while (this.tags.size() < ITEMS_LIST_LIMIT && index < tag.length) {
            this.tags.add(tag[index]);
            index++;
        }
        if (index < tag.length) {
            InstabugSDKLogger.w(Constants.LOG_TAG, "Some tags were ignored. Maximum allowed tags count reached. Please note that you can add up to " + ITEMS_LIST_LIMIT + " tags.");
        }
    }

    public void addTag(ArrayList<String> tags) {
        if (tags == null) return;
        int index = 0;
        while (this.tags.size() < ITEMS_LIST_LIMIT && index < tags.size()) {
            this.tags.add(tags.get(index));
            index++;
        }
        if (index < tags.size()) {
            InstabugSDKLogger.w(Constants.LOG_TAG, "Some tags were ignored. Maximum allowed tags count reached. Please note that you can add up to " + ITEMS_LIST_LIMIT + " tags.");
        }
    }

    public void addFileAttachment(@NonNull Uri fileUri, @NonNull String fileNameWithExtension) {
        if (fileAttachments.size() < FILES_LIST_LIMIT) {
            fileAttachments.put(fileUri, fileNameWithExtension);
        } else {
            InstabugSDKLogger.w(Constants.LOG_TAG, "File was ignored. Maximum allowed files count reached. Please note that you can add up to " + ITEMS_LIST_LIMIT + " files.");
        }
    }

    public void addFileAttachment(@NonNull byte[] data, @NonNull String fileNameWithExtension) {
        if (fileAttachments.size() >= FILES_LIST_LIMIT) {
            InstabugSDKLogger.w(Constants.LOG_TAG, "File was ignored. Maximum allowed files count reached. Please note that you can add up to " + ITEMS_LIST_LIMIT + " files.");
            return;
        }

        Context context = Instabug.getApplicationContext();

        if (context != null) {
            if (AttachmentsUtility.isValidSize(data.length, AttachmentsUtility.MAX_FILE_SIZE_IN_MB)) {
                Uri uriFromBytes = AttachmentsUtility.getUriFromBytes(context, data, fileNameWithExtension);
                if (uriFromBytes != null) {
                    fileAttachments.put(uriFromBytes, fileNameWithExtension);
                }
            }
        }
    }

    /**
     * Sets user attribute to overwrite it's value or create a new one if it doesn't exist.
     *
     * @param key   the attribute
     * @param value the value
     */
    public void setUserAttribute(@NonNull String key, String value) {
        if (userAttributes.size() >= ITEMS_LIST_LIMIT) {
            InstabugSDKLogger.w(Constants.LOG_TAG, "User attribute was ignored. Maximum allowed attributes count reached. Please note that you can add up to " + ITEMS_LIST_LIMIT + " user attributes.");
            return;
        }
        userAttributes.put(key, value);
    }

    /**
     * Sets user attribute to append on old values
     *
     * @param userAttributes All new userAttributes to be added
     */
    public void addUserAttributes(HashMap<String, String> userAttributes) {
        if (userAttributes == null) return;

        Iterator<Map.Entry<String, String>> iterator = userAttributes.entrySet().iterator();
        while (this.userAttributes.size() < ITEMS_LIST_LIMIT && iterator.hasNext()) {
            Map.Entry<String, String> entry = iterator.next();
            if (entry != null) {
                this.userAttributes.put(entry.getKey(), entry.getValue());
            }
        }

        if (iterator.hasNext()) {
            InstabugSDKLogger.w(Constants.LOG_TAG, "Some attributes were ignored. Maximum allowed attributes count reached. Please note that you can add up to " + ITEMS_LIST_LIMIT + " user attributes.");
        }
    }

    public void appendToConsoleLogs(String logMessage) {

        if (consoleLogs.size() >= ITEMS_LIST_LIMIT) {
            InstabugSDKLogger.w(Constants.LOG_TAG, "Console log was ignored. Maximum allowed logs count reached. Please note that you can add up to " + ITEMS_LIST_LIMIT + " log messages.");
            return;
        }

        ConsoleLog consoleLog = new ConsoleLog();
        consoleLog.setMessage(logMessage);
        consoleLog.setTimeStamp(System.currentTimeMillis());
        consoleLogs.add(consoleLog);
    }

    public void logVerbose(String logMessage) {
        if (InstabugCore.getFeatureState(IBGFeature.INSTABUG_LOGS)
                == Feature.State.ENABLED) {
            InstabugLog.v(logMessage);
            instabugLogs.add(InstabugLog.LogMessage.generateLogMessage(logMessage, InstabugLog.LogLevel.V));
        }
    }

    public void logDebug(String logMessage) {
        if (InstabugCore.getFeatureState(IBGFeature.INSTABUG_LOGS)
                == Feature.State.ENABLED) {
            InstabugLog.d(logMessage);
            instabugLogs.add(InstabugLog.LogMessage.generateLogMessage(logMessage, InstabugLog.LogLevel.D));
        }
    }

    public void logInfo(String logMessage) {
        if (InstabugCore.getFeatureState(IBGFeature.INSTABUG_LOGS)
                == Feature.State.ENABLED) {
            InstabugLog.i(logMessage);
            instabugLogs.add(InstabugLog.LogMessage.generateLogMessage(logMessage, InstabugLog.LogLevel.I));
        }
    }

    public void logWarn(String logMessage) {
        if (InstabugCore.getFeatureState(IBGFeature.INSTABUG_LOGS)
                == Feature.State.ENABLED) {
            InstabugLog.w(logMessage);
            instabugLogs.add(InstabugLog.LogMessage.generateLogMessage(logMessage, InstabugLog.LogLevel.W));
        }
    }

    public void logError(String logMessage) {
        if (InstabugCore.getFeatureState(IBGFeature.INSTABUG_LOGS)
                == Feature.State.ENABLED) {
            InstabugLog.e(logMessage);
            instabugLogs.add(InstabugLog.LogMessage.generateLogMessage(logMessage, InstabugLog.LogLevel.E));
        }
    }

    public List<String> getTags() {
        return tags;
    }

    public ArrayList<ConsoleLog> getConsoleLog() {
        return consoleLogs;
    }

    @Nullable
    public String getUserData() {
        return userData;
    }

    /**
     * Adds specific user data that you need to be added to the reports
     *
     * @param userData string representing user data
     */
    public void setUserData(@NonNull String userData) {
        this.userData = StringUtility.trimString(userData, 1000);
    }

    public HashMap<String, String> getUserAttributes() {
        return userAttributes;
    }

    public HashMap<Uri, String> getFileAttachments() {
        return fileAttachments;
    }

    public interface OnReportCreatedListener {
        void onReportCreated(Report report);
    }


    public void addConsoleLogs(ArrayList<ConsoleLog> consoleLogArrayList) {

        if (consoleLogArrayList == null) return;
        int index = 0;
        while (this.consoleLogs.size() <= ITEMS_LIST_LIMIT && index < consoleLogArrayList.size()) {
            this.consoleLogs.add(consoleLogArrayList.get(index));
            index++;
        }
        if (index < consoleLogArrayList.size()) {
            InstabugSDKLogger.w(Constants.LOG_TAG, "Some console logs were ignored. Maximum allowed logs count reached. Please note that you can add up to " + ITEMS_LIST_LIMIT + " log messages.");
        }
    }
}
