package com.instabug.library.logging.disklogs

import android.content.Context
import androidx.annotation.VisibleForTesting
import androidx.annotation.WorkerThread
import com.instabug.library.InstabugState
import com.instabug.library.InstabugStateProvider
import com.instabug.library.internal.resolver.LoggingSettingResolver
import com.instabug.library.internal.storage.DiskUtils
import com.instabug.library.model.LogData
import com.instabug.library.model.LogDescriptor
import com.instabug.library.model.LoggingSettings
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.threading.PoolProvider
import java.lang.ref.WeakReference

class IBGLoggingThread(val context: Context) : Thread() {

    private val TAG = "IBGDiskLoggingThread"

    private val END_SESSION = "End-session"
    private var loggingInterval: Long =
        LoggingSettingResolver.getInstance().loggingSettings?.flushInterval
            ?: LoggingSettings.DEFAULT_FLUSH_INTERVAL_SECONDS * 1000L
    private val contextReference: WeakReference<Context> = WeakReference(context)
    private val loggingFileProvider = LoggingFileProvider(context)

    @Volatile
    @VisibleForTesting
    var logsStringBuilder = StringBuilder()
    var wasInterrupted: Boolean = false

    val loggingExecutor = PoolProvider.getSingleThreadExecutor("LoggingExecutor")

    init {
        start()
    }

    override fun run() {

        name = "IBGLoggingThread"

        while (LoggingSettingResolver.getInstance().loggingSettings?.level != LoggingSettings.LogLevels.NO_LOGS && !wasInterrupted) {

            try {
                Thread.sleep(loggingInterval)
            } catch (e: InterruptedException) {
                InstabugSDKLogger.v(TAG, "IBGDiskLoggingThread was interrupted")
            }

            if (logsStringBuilder.isNotEmpty()) {
                loggingExecutor.execute {
                    writeLogs()
                }
            }
        }
    }

    fun addLog(tag: String, msg: String, currentThread: String, timeStamp: Long) {
        val log = LogData.Builder()
            .setTag(tag)
            .setMessage(trimLogMessageIfNeeded(msg))
            .setCurrentThreadName(currentThread)
            .setTimestamp(timeStamp)
            .build()

        logsStringBuilder.append(log.toString())

        flushToDiskIfNeeded()
    }

    @VisibleForTesting
    fun flushToDiskIfNeeded() {
        if (hasExceededFlushCharLimit()) {
            writeLogs()
        }
    }

    @VisibleForTesting
    fun hasExceededFlushCharLimit() : Boolean {
        return logsStringBuilder.length >= LoggingSettingResolver.getInstance().loggingSettings?.flushCharLimit ?: LoggingSettings.DEFAULT_FLUSH_CHAR_LIMIT
    }

    @VisibleForTesting
    fun trimLogMessageIfNeeded(msg: String): String {
        val limit = LoggingSettingResolver.getInstance().loggingSettings?.singleLogLimit
            ?: LoggingSettings.DEFAULT_SINGLE_LOG_LIMIT
        if (msg.length > limit) {
            val difference = msg.length - limit
            val msgBuilder = StringBuilder(msg)
            msgBuilder.delete(limit.toInt(), msg.length)
            msgBuilder.append("...$difference")
            return msgBuilder.toString()
        }
        return msg
    }

    @WorkerThread
    @VisibleForTesting
    fun writeLogs() {
        if (InstabugStateProvider.getInstance().state != InstabugState.DISABLED) {
            val todayFile = loggingFileProvider.getTodayFile()
            val context = contextReference.get()
            if (todayFile != null && context != null) {
                DiskUtils.with(context)
                    .writeOperation(
                        WriteLogDiskOperator(
                            todayFile,
                            logsStringBuilder.toString()
                        )
                    )
                    .execute()
                logsStringBuilder.setLength(0)
                loggingFileProvider.trimDirectoryIfNeeded()
            }
        } else {
            logsStringBuilder.setLength(0)
        }
    }

    fun logSessionDescriptor(sessionDescriptor: LogDescriptor) {
        logsStringBuilder.append(sessionDescriptor)
    }

    fun logEndSession(timestamp: Long) {
        addLog(tag = "", msg = END_SESSION, timeStamp = timestamp, currentThread = "")
    }

    override fun interrupt() {
        wasInterrupted = true;
        super.interrupt()
    }


}