package com.instabug.library.internal.sharedpreferences

import android.content.SharedPreferences
import androidx.annotation.VisibleForTesting
import java.lang.ref.WeakReference
import kotlin.properties.ReadWriteProperty
import kotlin.reflect.KProperty

abstract class WeakPreferencesProperty<T>(
    protected val key: String,
    private val propertyAccessStrategy: ReadWriteStrategyPreferenceProperty.ReadWriteStrategy<T?>
) : ReadWriteProperty<Any?, T?> {

    @Volatile
    @VisibleForTesting
    var value: WeakReference<T>? = null

    @Volatile
    @VisibleForTesting
    var valueAbsentState: Boolean = false

    abstract val preference: SharedPreferences?

    override fun getValue(thisRef: Any?, property: KProperty<*>): T? = synchronized(this) {
        if (valueAbsentState) {
            return null
        }
        value?.get() ?: let {
            val persistedValue = with(propertyAccessStrategy) { preference?.get(key, null) }
            if (persistedValue == null) {
                valueAbsentState = true
            }
            value = WeakReference(persistedValue)
            persistedValue
        }
    }
    override fun setValue(thisRef: Any?, property: KProperty<*>, value: T?) {
        synchronized(this) {
            this.value = WeakReference(value)
            valueAbsentState = value == null
        }
        persistValue(value)
    }
    protected fun persistValue(value: T?) {
        with(propertyAccessStrategy) {
            preference?.edit()?.put(key, value)?.apply()
        }
    }
    open fun clear() {
        value = null
        clearPersistedValue()
        valueAbsentState = false
    }
    protected fun clearPersistedValue() {
        preference?.edit()?.remove(key)?.apply()
    }
}
