package com.instabug.library.diagnostics.network

import com.instabug.library.Constants
import com.instabug.library.diagnostics.customtraces.di.CustomTracesServiceLocator
import com.instabug.library.diagnostics.di.DiagnosticsServiceLocator
import com.instabug.library.diagnostics.mappers.DiagnosticsRequestParam
import com.instabug.library.diagnostics.mappers.RequestParameterResolver
import com.instabug.library.diagnostics.mappers.isEmpty
import com.instabug.library.diagnostics.mappers.param
import com.instabug.library.diagnostics.mappers.isEmpty
import com.instabug.library.diagnostics.mappers.param
import com.instabug.library.diagnostics.nonfatals.networking.NonFatalSyncException
import com.instabug.library.diagnostics.sdkEvents.di.SDKEventsServiceLocator
import com.instabug.library.networkv2.INetworkManager
import com.instabug.library.networkv2.NetworkManager
import com.instabug.library.networkv2.RequestResponse
import com.instabug.library.networkv2.request.Request
import com.instabug.library.networkv2.request.RequestMethod
import com.instabug.library.networkv2.request.RequestParameter
import com.instabug.library.networkv2.request.RequestType
import com.instabug.library.settings.SettingsManager
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.TimeUtils.MINUTE
import com.instabug.library.diagnostics.nonfatals.di.ServiceLocator as NonfatalsServiceLocator

/**
 * Created by Shannan on 06/09/2021.
 */
class IBGDiagnosticsSyncManagerImpl : IBGDiagnosticsSyncManager {

    private val networkManager: INetworkManager
        get() = DiagnosticsServiceLocator.getNetworkManager()
    private val settingsManager: SettingsManager
        get() = SettingsManager.getInstance()
    private val requestParamMappers: List<RequestParameterResolver<*>> = listOf(
        NonfatalsServiceLocator.getNonFatalsRequestParamResolver(),
        SDKEventsServiceLocator.sdkEventRequestParameterResolver,
        CustomTracesServiceLocator.getCustomTracesRequestParamResolver()
    )

    override fun sync(callback: IBGDiagnosticsResponseCallback?) {
        requestParamMappers
            .takeIf { isSyncIntervalPassed() }
            ?.also {
                InstabugSDKLogger.d(
                    Constants.LOG_TAG,
                    "Sync Interval Passed syncing some exceptions"
                )
            }?.runCatching { syncOnIntervalPassed(this, callback) }
            ?.onFailure { callback?.onFailed(it) }
    }

    private fun syncOnIntervalPassed(
        mappers: List<RequestParameterResolver<*>>,
        callback: IBGDiagnosticsResponseCallback?
    ) {
        var requestParams = listOf<RequestParameter<*>>()
        var canRequest = false
        mappers.map { it.toRequestParameter() }.run {
            requestParams = map { it.param }
            canRequest = any { it.isEmpty.not() }
        }
        if (canRequest) doRequest(constructRequest(requestParams), callback)
    }

    override fun setOnDoRequestListener(onDoRequestListener: NetworkManager.OnDoRequestListener?) {
        networkManager.onDoRequestListener = onDoRequestListener
    }

    private fun doRequest(
        request: Request?,
        callback: IBGDiagnosticsResponseCallback?
    ) = request
        ?.let {
            networkManager.doRequestOnSameThread(
                RequestType.NORMAL,
                request,
                getRequestCallbacks(callback, request.requestBodyParameters)
            )
        } ?: callback?.onFailed(NonFatalSyncException("Request object can't be null"))

    private fun getRequestCallbacks(
        callback: IBGDiagnosticsResponseCallback?,
        parameters: List<RequestParameter<*>>
    ) = object : Request.Callbacks<RequestResponse, Throwable> {
        override fun onSucceeded(response: RequestResponse?) {
            callback?.onSucceeded(parameters)
        }

        override fun onFailed(error: Throwable) {
            callback?.onFailed(error)
        }
    }

    private fun constructRequest(requestParams: List<RequestParameter<*>>): Request = Request
        .Builder()
        .url(DIAGNOSTICS_URL)
        .method(RequestMethod.POST)
        .apply { requestParams.forEach(::addParameter) }
        .shorten(false)
        .hasUuid(false)
        .disableDefaultParameters(true)
        .build()

    private fun isSyncIntervalPassed(): Boolean = settingsManager.diagnosticsSyncInterval.let {
        (System.currentTimeMillis() - settingsManager.diagnosticsLastSyncTime) >= (it * MINUTE)
    }
}