package com.instabug.library.core.plugin;

import android.annotation.SuppressLint;
import android.content.Context;

import androidx.annotation.NonNull;

import com.instabug.library.Constants;
import com.instabug.library.PresentationManager;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.internal.dataretention.DisposalPolicy;
import com.instabug.library.sessionV3.di.IBGSessionServiceLocator;
import com.instabug.library.sessionV3.providers.FeatureSessionDataController;
import com.instabug.library.sessionV3.providers.FeatureSessionLazyDataProvider;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.InstabugSDKLogger;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Locale;

/**
 * Created by tarek on 3/25/17.
 */

public class PluginsManager {
    private static final Object lock = new Object();
    private volatile static List<Plugin> PLUGINS;

    /**
     * Init the PluginsManager to get and init the defined plugins.
     */
    public static void init(Context ctx) {
        InstabugSDKLogger.v(Constants.LOG_TAG, "Initializing plugins");
        synchronized (lock) {
            if (PLUGINS == null) {
                PLUGINS = new ArrayList<>();
                String[] plugins = {
                        "com.instabug.compose.ComposePlugin",
                        "com.instabug.crash.CrashPlugin",
                        "com.instabug.survey.SurveyPlugin", "com.instabug.chat.ChatPlugin",
                        "com.instabug.bug.BugPlugin", "com.instabug.featuresrequest.FeaturesRequestPlugin",
                        "com.instabug.apm.APMPlugin"
                };
                for (String pluginClassPath : plugins) {
                    try {
                        Plugin plugin = (Plugin) Class.forName(pluginClassPath).newInstance();
                        plugin.init(ctx);
                        PLUGINS.add(plugin);
                        InstabugSDKLogger.d(Constants.LOG_TAG, "pluginClassPath: " +
                                pluginClassPath);

                    } catch (ClassNotFoundException e) {
                        InstabugSDKLogger.v(Constants.LOG_TAG, "Can't load " + pluginClassPath);
                    } catch (InstantiationException e) {
                        InstabugSDKLogger.e(Constants.LOG_TAG, "InstantiationException Can't get: " + pluginClassPath);
                    } catch (IllegalAccessException e) {
                        InstabugSDKLogger.e(Constants.LOG_TAG, "IllegalAccessException Can't get: " + pluginClassPath);
                    }
                }
            }
        }
    }

    public static void wake() {
        synchronized (lock) {
            if (!arePluginsInitialized("wake()")) {
                return;
            }
            for (Plugin plugin : PLUGINS) {
                plugin.wakeIfPossible();
            }
        }
    }

    public static void sleep() {
        synchronized (lock) {
            if (!arePluginsInitialized("sleep()")) {
                return;
            }
            for (Plugin plugin : PLUGINS) {
                plugin.sleepIfPossible();
            }
            PresentationManager.release();
        }
    }

    /**
     * Sleep the plugins.
     */
    public static void stopPlugins() {
        synchronized (lock) {
            if (!arePluginsInitialized("stopPlugins()")) {
                return;
            }
            for (Plugin plugin : PLUGINS) {
                plugin.stopIfPossible();
            }
        }
    }

    /**
     * Wake the plugins.
     */

    public static void startPlugins(Context context) {
        synchronized (lock) {
            if (!arePluginsInitialized("startPlugins()")) {
                return;
            }
            for (Plugin plugin : PLUGINS) {
                plugin.startIfPossible(context);
            }
        }
    }


    public static long getLastActivityTime() {
        synchronized (lock) {
            long lastActivityTime = 0L;
            if (!arePluginsInitialized("getLastActivityTime()")) {
                return lastActivityTime;
            }
            for (Plugin plugin : PLUGINS) {
                long pluginLastActivityTime = plugin.getLastActivityTime();
                if (pluginLastActivityTime > lastActivityTime) {
                    lastActivityTime = pluginLastActivityTime;
                }
            }
            return lastActivityTime;
        }
    }

    @NonNull
    public static ArrayList<PluginPromptOption> getPluginsPromptOptions() {
        synchronized (lock) {
            ArrayList<PluginPromptOption> allPluginsPromptOptions = new ArrayList<>();
            if (!arePluginsInitialized("getPluginsPromptOptions()")) {
                return allPluginsPromptOptions;
            }
            for (Plugin plugin : PLUGINS) {
                ArrayList<PluginPromptOption> pluginPromptOptions = plugin.getPromptOptions();
                if (pluginPromptOptions != null) {
                    allPluginsPromptOptions.addAll(pluginPromptOptions);
                }
            }
            sortPluginsPromptOptions(allPluginsPromptOptions);
            return allPluginsPromptOptions;
        }
    }

    /**
     * A method to return all available options regardless the prompt option,
     *
     * @return
     */
    @NonNull
    public static ArrayList<PluginPromptOption> getPluginOptions(boolean ignoreBaseFeature) {
        synchronized (lock) {
            InstabugSDKLogger.d(Constants.LOG_TAG, "[PluginsManager#getPluginOptions] Getting plugin options");
            ArrayList<PluginPromptOption> allPluginsPromptOptions = new ArrayList<>();
            if (!arePluginsInitialized("getPluginOptions()")) {
                return allPluginsPromptOptions;
            }
            for (Plugin plugin : PLUGINS) {
                ArrayList<PluginPromptOption> pluginPromptOptions = plugin.getPluginOptions(ignoreBaseFeature);
                if (pluginPromptOptions != null) {
                    allPluginsPromptOptions.addAll(pluginPromptOptions);
                }
            }

            sortPluginsPromptOptions(allPluginsPromptOptions);
            InstabugSDKLogger.d(Constants.LOG_TAG, "[PluginsManager#getPluginOptions] Found plugin options: " + allPluginsPromptOptions);
            return allPluginsPromptOptions;
        }
    }


    public static void initPluginsPromptOptionAvailability() {
        synchronized (lock) {
            if (!arePluginsInitialized("initPluginsPromptOptionAvailability()")) {
                return;
            }
            for (Plugin plugin : PLUGINS) {
                plugin.initDefaultPromptOptionAvailabilityState();
            }
        }
    }


    public static void notifyPluginsLocaleChanged(Locale oldLocale, Locale newLocale) {
        synchronized (lock) {
            if (!arePluginsInitialized("notifyPluginsLocaleChanged()")) {
                return;
            }
            for (Plugin plugin : PLUGINS) {
                plugin.onLocaleChanged(oldLocale, newLocale);
            }
        }
    }

    private static void sortPluginsPromptOptions(ArrayList<PluginPromptOption>
                                                         pluginPromptOptions) {
        Collections.sort(pluginPromptOptions, new PluginPromptOption.CustomComparator());
    }

    @NonNull
    @Deprecated

    public static List<DisposalPolicy> getDataDisposalPolicies() {
        synchronized (lock) {
            List<DisposalPolicy> disposalPolicies = new ArrayList<>();
            if (arePluginsInitialized("getDataDisposalPolicies()")) {
                for (Plugin plugin : PLUGINS) {
                    disposalPolicies.addAll(plugin.getDataDisposalPolicies());
                }
                return disposalPolicies;
            }
            return disposalPolicies;
        }
    }


    @SuppressLint("THREAD_SAFETY_VIOLATION")
    public static boolean isForegroundBusy() {
        synchronized (lock) {
            return foregroundBusy();
        }
    }

    /*same function as above but without synchronized*/
    public static boolean foregroundBusy() {
        try {
            boolean isAPluginInForeground = false;
            if (!arePluginsInitialized("isForegroundBusy()")) {
                return checkForegroundBusy();
            }
            List<Plugin> plugins = PLUGINS;
            for (Plugin plugin : plugins) {
                if (plugin.getState() != Plugin.STATE_BACKGROUND) {
                    isAPluginInForeground = true;
                }
            }
            return isAPluginInForeground || checkForegroundBusy();
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error in isForegroundBusy", e);
            return false;
        }
    }

    public static boolean checkForegroundBusy() {
        return SettingsManager.getInstance().isPromptOptionsScreenShown()
                || SettingsManager.getInstance().isRequestPermissionScreenShown()
                || SettingsManager.getInstance().isOnboardingShowing();
    }

    @SuppressLint({"ERADICATE_RETURN_NOT_NULLABLE"})
    public static Plugin getXPlugin(Class x) {
        synchronized (lock) {
            if (!arePluginsInitialized("getXPlugin()")) {
                return null;
            }
            for (Plugin plugin : PLUGINS) {
                if (x.isInstance(plugin)) {
                    return plugin;
                }
            }
            return null;
        }
    }

    /**
     * A method to return all available  features session data controllers
     * returns empty list if plugins are not initialized
     * return empty list if no plugin implements FeatureSessionDataControllerHost
     *
     * @return List<FeatureSessionDataController>
     * @see com.instabug.library.sessionV3.providers.FeatureSessionDataControllerHost
     * @see FeatureSessionDataController
     */
    @NonNull
    public static List<FeatureSessionDataController> getFeaturesSessionDataControllers() {
        synchronized (lock) {
            try {
                if (!arePluginsInitialized("getFeaturesSessionDataControllers()"))
                    return new ArrayList<>();
                PluginsExtractor extractor = IBGSessionServiceLocator.getPluginsExtractor();
                return extractor.extractSessionDataControllers(PLUGINS);
            } catch (Throwable throwable) {
                IBGDiagnostics.reportNonFatalAndLog(throwable, "couldn't getFeaturesSessionDataControllers" + throwable.getMessage(), Constants.LOG_TAG);
            }
            return new ArrayList<>();

        }
    }

    @NonNull
    public static List<FeatureSessionLazyDataProvider> getFeaturesSessionLazyDataProvider() {
        synchronized (lock) {
            try {
                if (!arePluginsInitialized("getFeaturesSessionLazyDataProvider()"))
                    return new ArrayList<>();
                PluginsExtractor extractor = IBGSessionServiceLocator.getPluginsExtractor();
                return extractor.extractSessionDataLazyProviders(PLUGINS);
            } catch (Throwable throwable) {
                IBGDiagnostics.reportNonFatalAndLog(throwable, "couldn't getFeaturesSessionLazyDataProvider" + throwable.getMessage(), Constants.LOG_TAG);
            }
            return new ArrayList<>();
        }
    }

    private static boolean arePluginsInitialized(String callingMethodName) {
        if (PLUGINS == null) {
            logPluginsNotInitializedError(callingMethodName);
            return false;
        }
        return true;
    }

    private static void logPluginsNotInitializedError(String callingMethodName) {
        InstabugSDKLogger.e(Constants.LOG_TAG,
                "PluginsManager." + callingMethodName + " was called before PluginsManager.init() was " +
                        "called");
    }
}