package com.instabug.library.core;

import static com.instabug.library.screenshot.instacapture.ScreenShotType.INITIAL;

import android.app.Activity;
import android.content.Context;
import android.graphics.Bitmap;
import android.net.Uri;
import android.widget.Toast;

import androidx.annotation.NonNull;

import com.instabug.library.Constants;
import com.instabug.library.Instabug;
import com.instabug.library.R;
import com.instabug.library.screenshot.ScreenshotCaptor;
import com.instabug.library.screenshot.ScreenshotProvider;
import com.instabug.library.screenshot.instacapture.ScreenshotRequest;
import com.instabug.library.screenshot.instacapture.ScreenshotRequestArgs;
import com.instabug.library.tracking.InstabugInternalTrackingDelegate;
import com.instabug.library.util.BitmapUtils;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.LocaleUtils;
import com.instabug.library.util.memory.MemoryUtils;

/**
 * Created by mohamedzakaria on 6/21/17.
 */

public class InitialScreenshotHelper {


    public static void captureScreenshot(@NonNull final InitialScreenshotCapturingListener initialScreenshotCapturingListenr) {
        InstabugSDKLogger.d(Constants.LOG_TAG, "[InitialScreenshotHelper#captureScreenshot] Starting in screenshot capturing process");
        final Activity activity = InstabugInternalTrackingDelegate.getInstance().getTargetActivity();
        if (activity == null) {
            InstabugSDKLogger.d(Constants.LOG_TAG, "[InitialScreenshotHelper#captureScreenshot] No target activity found");
            initialScreenshotCapturingListenr.onScreenshotCapturingFailed(new Throwable("Target Activity not found"));
            return;
        }
        //don't try to take a screenshot if the memory already low
        if (MemoryUtils.isLowMemory(activity)) {
            InstabugSDKLogger.e(Constants.LOG_TAG,
                    "Couldn't take initial screenshot due to low memory");
            initialScreenshotCapturingListenr.onScreenshotCapturingFailed(new Throwable("Your activity is currently in low memory"));

            String message = LocaleUtils.getLocaleStringResource(InstabugCore.getLocale(activity),
                    R.string.instabug_str_capturing_screenshot_error, activity);
            Toast.makeText(activity, message, Toast.LENGTH_SHORT).show();
            return;
        }
        ScreenshotProvider.INSTANCE.capture(createInitialScreenshotRequest(initialScreenshotCapturingListenr, activity));
    }

    @NonNull
    private static ScreenshotRequest createInitialScreenshotRequest(@NonNull InitialScreenshotCapturingListener initialScreenshotCapturingListenr, Activity activity) {
        return ScreenshotRequest.createScreenshotRequest(
                new ScreenshotRequestArgs(
                        INITIAL,
                        activity,
                        createCapturingListener(initialScreenshotCapturingListenr, activity)
                )
        );
    }

    @NonNull
    private static ScreenshotCaptor.CapturingCallback createCapturingListener(@NonNull InitialScreenshotCapturingListener initialScreenshotCapturingListenr, Activity activity) {
        return new ScreenshotCaptor.CapturingCallback() {
            @Override
            public void onCapturingSuccess(@NonNull Bitmap bitmap) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "[InitialScreenshotHelper#createCapturingListener] Capturing succeeded.");
                saveBitmap(bitmap, activity, initialScreenshotCapturingListenr);
            }

            @Override
            public void onCapturingFailure(@NonNull Throwable throwable) {
                InstabugSDKLogger.e(Constants.LOG_TAG
                        , "initial screenshot capturing got error: " + throwable.getMessage());
                initialScreenshotCapturingListenr.onScreenshotCapturingFailed(throwable);
            }
        };
    }

    private static void saveBitmap(Bitmap bitmap, Activity activity, final InitialScreenshotCapturingListener initialScreenshotCapturingListenr) {
        Context context = Instabug.getApplicationContext();
        if (context != null) {
            BitmapUtils.saveBitmap(bitmap, activity, new
                    BitmapUtils.OnSaveBitmapCallback() {

                        @Override
                        public void onSuccess(Uri screenshotUri) {
                            InstabugSDKLogger.d(Constants.LOG_TAG, "[InitialScreenshotHelper#saveBitmap] Saving bitmap succeeded.");
                            initialScreenshotCapturingListenr
                                    .onScreenshotCapturedSuccessfully(screenshotUri);
                        }

                        @Override
                        public void onError(Throwable throwable) {
                            InstabugSDKLogger.e(Constants.LOG_TAG, "initial screenshot capturing got error: " + throwable.getMessage());
                            initialScreenshotCapturingListenr.onScreenshotCapturingFailed(throwable);
                        }
                    });
        }
    }


    public interface InitialScreenshotCapturingListener {
        void onScreenshotCapturedSuccessfully(Uri screenshotUri);

        void onScreenshotCapturingFailed(Throwable throwable);
    }
}
