package com.instabug.library.visualusersteps;

import android.annotation.SuppressLint;
import android.content.Context;
import android.graphics.Bitmap;
import android.net.Uri;
import android.view.View;
import android.view.ViewParent;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.core.InstabugCore;
import com.instabug.library.interactionstracking.UINodeTransformer;
import com.instabug.library.internal.servicelocator.CoreServiceLocator;
import com.instabug.library.internal.storage.AttachmentManager;
import com.instabug.library.internal.storage.ProcessedUri;
import com.instabug.library.screenshot.instacapture.ViewsFilter;
import com.instabug.library.util.BitmapUtils;
import com.instabug.library.util.DiskUtils;
import com.instabug.library.util.FileUtils;
import com.instabug.library.util.threading.PoolProvider;

import java.io.File;
import java.util.ArrayList;

import io.reactivexport.Observable;

/**
 * @author hossam.
 */

public class VisualUserStepsHelper {

    /**
     * Gets uri for for the zipped file that contains
     * the visual user steps screenshots to be attached as zip file to reports
     *
     * @param context AppContext
     * @param id
     * @return Uri for the zipped file that contains
     * the visual user steps screenshots to be attached as zip file to reports
     */
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public static ProcessedUri getVisualUserStepsFile(Context context, String id) {
        final File directory = CoreServiceLocator.getReproScreenshotsCacheDir().getCurrentSpanDirectory();
        return getVisualUserStepsFile(context, id, directory);
    }

    public static ProcessedUri getVisualUserStepsFile(Context context, String id, File screenshotsDir) {
        boolean isZipUriEncrypted = false;
        ArrayList<File> files = DiskUtils.listFilesInDirectory(screenshotsDir);

        for (File file : files) {
            // Decrypt the file only if it was encrypted
            if (FileUtils.isEncryptedFile(file.getPath())) {
                InstabugCore.decrypt(file.getPath());
            }
        }

        ArrayList<File> decryptedFiles = PoolProvider.getFilesEncryptionExecutor().executeAndGet(() -> DiskUtils.listFilesInDirectory(screenshotsDir));
        Uri zipUri = null;
        if (decryptedFiles != null) {
            zipUri = DiskUtils.zipFiles(context,
                    VisualUserStepsProvider.STEPS_COMPRESSED_FILE_NAME + id, decryptedFiles);
            if (zipUri != null && zipUri.getPath() != null) {
                isZipUriEncrypted = InstabugCore.encrypt(zipUri.getPath());
            }
        }

        files = PoolProvider.getFilesEncryptionExecutor().executeAndGet(() -> DiskUtils.listFilesInDirectory(screenshotsDir));
        if (files != null) {
            for (File file : files) {
                // Only re-encrypt the decrypted files
                if (!FileUtils.isEncryptedFile(file.getPath())) {
                    InstabugCore.encryptBeforeMarshmallow(file.getPath());
                }
            }
        }
        return new ProcessedUri(zipUri, isZipUriEncrypted);
    }

    public static File getVisualUserStepsDirectory(@Nullable Context context) {
        return AttachmentManager.getNewDirectory(context, VisualUserStepsProvider
                .STEPS_DIRECTORY_NAME);
    }

    public static File getVisualUserStepsInternalDirectory(@Nullable Context context) {
        return AttachmentManager.getNewInternalDirectory(context, VisualUserStepsProvider
                .STEPS_DIRECTORY_NAME);
    }

    @NonNull
    public static Observable<ProcessedUri> getVisualUserStepsFileObservable(final Context context,
                                                                            final String id) {
        return Observable.fromCallable(() -> getVisualUserStepsFile(context, id));
    }

    public static ArrayList<VisualUserStep> fetchSteps() {
        return CoreServiceLocator.getReproStepsProxy().fetch();
    }

    public static Bitmap decryptBitmap(String path) {
        // At this point, the file is decrepted
        return BitmapUtils.decryptBitmap(path.replace(FileUtils.FLAG_ENCRYPTED, ""));
    }

    public static void removeScreenshotId(String screenshotUri) {
        CoreServiceLocator.getReproStepsProxy().removeScreenshotId(screenshotUri);
    }

    /**
     * @param view the touched view
     * @return true if the view is private or in a private view
     * or should be masked using the {@link com.instabug.library.MaskingType}
     */
    public static boolean isPrivateView(View view) {
        if (view == null) {
            return false;
        }
        ViewsFilter filter = CoreServiceLocator.getComposedViewsFilter();
        UINodeTransformer<View> transformer = CoreServiceLocator.getUiNodesTransformer();
        boolean shouldMaskView = filter.shouldMask(transformer.transform(view));
        return shouldMaskView || isInPrivateScope(view, filter, transformer);
    }

    /**
     * Check if the view is a child of a private ViewGroup
     *
     * @param view the touched view.
     * @return true if the {@param privateView} is private and is a parent of {@param view}
     */
    private static boolean isInPrivateScope(View view, ViewsFilter filter, UINodeTransformer<View> transformer) {
        if (view == null) {
            return false;
        }
        ViewParent parent = view.getParent();
        if (parent instanceof View) {
            if (filter.shouldMask(transformer.transform((View) parent))) {
                return true;
            }
            return isInPrivateScope((View) (parent), filter, transformer);
        }
        return false;
    }

    /**
     * Encrypts existing visual user steps
     */
    public static void encryptExistingSteps() {
        PoolProvider.postIOTask(() -> {
            File directory = CoreServiceLocator.getReproScreenshotsCacheDir().getCurrentSpanDirectory();
            if (directory != null) {
                ArrayList<File> files = DiskUtils.listFilesInDirectory(directory);
                for (File file : files) {
                    // Only re-encrypt the decrypted files
                    if (!FileUtils.isEncryptedFile(file.getPath())) {
                        InstabugCore.encryptBeforeMarshmallow(file.getPath());
                    }
                }
            }
        });

    }
}
