package com.instabug.library.tracking

import android.app.Activity
import androidx.annotation.IntDef
import androidx.fragment.app.DialogFragment
import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentActivity
import androidx.fragment.app.FragmentManager
import com.instabug.library.tracking.FragmentNode.LifecycleStage.Companion.Attached
import com.instabug.library.tracking.FragmentNode.LifecycleStage.Companion.Detached
import com.instabug.library.tracking.FragmentNode.LifecycleStage.Companion.Instantiated
import com.instabug.library.tracking.FragmentNode.LifecycleStage.Companion.Paused
import com.instabug.library.tracking.FragmentNode.LifecycleStage.Companion.Resumed
import com.instabug.library.tracking.FragmentNode.LifecycleStage.Companion.Started
import com.instabug.library.tracking.FragmentNode.LifecycleStage.Companion.Stopped
import com.instabug.library.tracking.FragmentNode.LifecycleStage.Companion.ViewCreated
import com.instabug.library.util.TimeUtils
import java.lang.ref.WeakReference
import java.util.LinkedList

/**
 * A contract definition of possible screen nodes in a tree representing
 * initialized screen components ([Activity], [Fragment], etc ..).
 */
interface ScreenNode {
    /**
     * The unique identifier of the node.
     */
    val id: Int

    /**
     * The actual simplified screen name that's being represented by the node.
     */
    val simpleName: String

    /**
     * The actual full screen name (including package) that's being represented by the node.
     */
    val fullName: String

    /**
     * Whether the component represented by the node is in an active state.
     */
    val isActive: Boolean

    /**
     * Whether the component represented by the node is visible by default (Useful for Fragments ViewPager).
     */
    val isVisible: Boolean

    /**
     * The moment of time in nanos in which the user has biased the component that's
     * being represented by the node over other possibly active components.
     */
    val userDefinitionTime: Long

    /**
     * The time in nanos when the node became active.
     */
    val activationTime: Long

    /**
     * Marks the node as active corresponding to some event indicating that
     * the represented component is being active.
     */
    fun activate()

    /**
     * Marks the node as inactive corresponding to some event indicating that
     * the represented component is being inactive.
     */
    fun deactivate()

    /**
     * Marks the node as biased over other siblings because of a bias the user made
     * to the component that the node represents.
     */
    fun defineByUser()
}

class SimpleScreenNode(
    override val id: Int,
    override val simpleName: String,
    override val fullName: String
) : ScreenNode {
    override var isActive: Boolean = false
    override val isVisible: Boolean = true
    override var activationTime: Long = -1
    override var userDefinitionTime: Long = -1

    override fun activate() {
        activationTime = TimeUtils.nanoTime()
        isActive = true
    }

    override fun deactivate() {
        isActive = false
    }

    override fun defineByUser() {
        userDefinitionTime = TimeUtils.nanoTime()
    }
}

/**
 * A contract defining the functionalities of a container (parent)
 * with multiple [ScreenNode] children
 */
interface ScreenChildrenOwner {
    /**
     * The children nodes currently held by this parent.
     */
    val children: List<ScreenNode>

    /**
     * Adds a child directly to this parent
     * @param child The [ScreenNode] to be added.
     */
    fun addChild(child: ScreenNode)

    /**
     * Searches for a direct child to this parent given its id.
     * @param childId the id of the child being searched for.
     * @return The [ScreenNode] corresponding to that child or null if a direct child doesn't exist.
     */
    fun findChild(childId: Int): ScreenNode?

    /**
     * Removes a direct child to this parent given its id.
     * @param childId the id of the child being removed.
     */
    fun removeChild(childId: Int)

    /**
     * A helper object for cross-tree operations.
     */
    object Helper {
        /**
         * Traverses the tree in a breadth-first approach starting from a specific
         * node as its starting (root) node
         * @param startNode the [ScreenChildrenOwner] node to start the search with
         * @param criteria the criteria of traversal (whether to visit a specific node or not), defaults to true (visit all).
         * @param action the action to perform on visiting a node.
         */
        fun traverseBreadth(
            startNode: ScreenChildrenOwner,
            criteria: (ScreenNode) -> Boolean = { true },
            action: (ScreenNode) -> Unit
        ) {
            val starterNodes = with(startNode) {
                (this as? ScreenNode)?.takeIf(criteria)?.let(::listOf)
                    ?: children.filter(criteria)
            }
            val examinationQueue = LinkedList(starterNodes)
            var examinationNode: ScreenNode
            while (examinationQueue.isNotEmpty()) {
                examinationNode = examinationQueue.pop()
                action(examinationNode)
                (examinationNode as? ScreenChildrenOwner)?.children
                    ?.filter(criteria)?.let(examinationQueue::addAll)
            }
        }

        /**
         * Adds a child by searching the entire tree for its parent given the parent's id.
         * @param startNode the node to start searching for the parent from.
         * @param child the child [ScreenNode] to add to the parent when found.
         * @param parentId the id of the parent to whom the child should be added.
         */
        fun addChild(startNode: ScreenChildrenOwner, child: ScreenNode, parentId: Int) {
            traverseBreadth(startNode) { examinationNode ->
                if (examinationNode.id == parentId) {
                    (examinationNode as? ScreenChildrenOwner)?.addChild(child)
                }
            }
        }
    }
}

class SimpleChildrenOwner : ScreenChildrenOwner {

    private val _children: LinkedHashMap<Int, ScreenNode> = linkedMapOf()

    override val children: List<ScreenNode>
        get() = _children.values.toList()

    override fun addChild(child: ScreenNode) {
        _children[child.id] = child
    }

    override fun findChild(childId: Int): ScreenNode? = _children[childId]

    override fun removeChild(childId: Int) {
        _children.remove(childId)
    }
}

/**
 * An extension to [ScreenChildrenOwner] contract that specifies specific attributes
 * to have as a natural parent to [Fragment]s.
 */
interface FragmentsScreenChildrenOwner : ScreenChildrenOwner {
    /**
     * The fragments manager of the parent.
     */
    val childrenManager: FragmentManager?

    /**
     * The observer that should be deployed to receive children's lifecycle callbacks.
     */
    val childrenObserver: FragmentManager.FragmentLifecycleCallbacks?

    /**
     * A helper object for fragment specific cross-tree operations.
     */
    object Helper {
        /**
         * Injects the parent observer to its fragments manager.
         * @param parent the [FragmentsScreenChildrenOwner] to operate on
         * @see FragmentsScreenChildrenOwner.childrenManager
         * @see FragmentsScreenChildrenOwner.childrenObserver
         */
        fun injectChildrenObserver(parent: FragmentsScreenChildrenOwner) {
            parent.childrenManager?.let { manager ->
                parent.childrenObserver?.let { observer ->
                    manager.registerFragmentLifecycleCallbacks(observer, false)
                }
            }
        }

        /**
         * Recursively removes the observers hocked to a specific fragments parent
         * and its children (if any is a fragments parent itself)
         * @param parent the [FragmentsScreenChildrenOwner] that the observer should be revoked from.
         */
        fun removeChildrenObserver(parent: FragmentsScreenChildrenOwner) {
            parent.children.forEach { child ->
                (child as? FragmentsScreenChildrenOwner)?.let { removeChildrenObserver(it) }
            }
            parent.childrenManager?.let { manager ->
                parent.childrenObserver?.let { observer ->
                    manager.unregisterFragmentLifecycleCallbacks(observer)
                }
            }
        }
    }
}

class ScreensTreeHandler : ScreenChildrenOwner by SimpleChildrenOwner()
abstract class FragmentsParentNode(
    childrenManager: FragmentManager?
) : ScreenNode, FragmentsScreenChildrenOwner, ScreenChildrenOwner {
    private val _childrenManager: WeakReference<FragmentManager>
    private val _childrenObserver: FragmentManager.FragmentLifecycleCallbacks?

    init {
        _childrenManager = WeakReference(childrenManager)
        _childrenObserver = childrenManager?.let { IBGFragmentLifecycleMonitor(this) }
    }

    final override val childrenManager: FragmentManager?
        get() = _childrenManager.get()
    final override val childrenObserver: FragmentManager.FragmentLifecycleCallbacks?
        get() = _childrenObserver
}

class ActivityNode(
    delegate: ScreenNode,
    childrenManager: FragmentManager?
) : FragmentsParentNode(childrenManager),
    ScreenChildrenOwner by SimpleChildrenOwner(),
    ScreenNode by delegate {

    object Factory {
        fun create(activity: Activity): ActivityNode = ActivityNode(
            delegate = SimpleScreenNode(
                activity.hashCode(),
                activity.javaClass.simpleName,
                activity.javaClass.name
            ),
            childrenManager = (activity as? FragmentActivity)?.supportFragmentManager
        )
    }
}

class FragmentNode(
    private val delegate: ScreenNode,
    val isDialogFragment: Boolean,
    private var parent: FragmentsScreenChildrenOwner?,
    private val fragmentRef: WeakReference<Fragment>,
    childrenManager: FragmentManager?
) : FragmentsParentNode(childrenManager),
    ScreenChildrenOwner by SimpleChildrenOwner(),
    ScreenNode by delegate {

    private var _lifecycleStage: Int = Instantiated

    override val isVisible: Boolean
        get() = fragmentRef.get()?.userVisibleHint ?: false

    val viewId: Int
        get() = fragmentRef.get()?.view?.hashCode() ?: -1

    val lifecycleStage: Int
        get() = _lifecycleStage

    override fun activate() {
        activateParent()
        delegate.activate()
    }

    fun updateLifecycleStage(newStage: Int) {
        _lifecycleStage = newStage
    }

    fun clean() {
        fragmentRef.clear()
        parent = null
    }

    private fun activateParent() {
        parent?.let { it as? ScreenNode }
            ?.takeUnless(ScreenNode::isActive)
            ?.activate()
    }

    object Factory {
        fun create(fragment: Fragment, parent: FragmentsScreenChildrenOwner) = FragmentNode(
            delegate = SimpleScreenNode(
                fragment.hashCode(),
                fragment.javaClass.simpleName,
                fragment.javaClass.name
            ),
            isDialogFragment = fragment is DialogFragment,
            parent = parent,
            fragmentRef = WeakReference(fragment),
            childrenManager = fragment.childFragmentManager
        )
    }

    @IntDef(Instantiated, Attached, ViewCreated, Started, Resumed, Paused, Stopped, Detached)
    annotation class LifecycleStage {
        companion object {
            const val Instantiated = 0x00
            const val Attached = 0x01
            const val ViewCreated = 0x02
            const val Started = 0x04
            const val Resumed = 0x08
            const val Paused = 0x10
            const val Stopped = 0x20
            const val Detached = 0x040
        }
    }
}

class ComposeNode(
    val delegate: ScreenNode,
    var parent: ScreenChildrenOwner?
) : ScreenNode by delegate {

    override fun activate() {
        activateParent()
        delegate.activate()
    }

    private fun activateParent() {
        parent?.let { it as? ScreenNode }
            ?.takeUnless(ScreenNode::isActive)
            ?.activate()
    }

    fun clean() {
        parent = null
    }

    object Factory {
        fun create(id: Int, simpleName: String, parent: ScreenChildrenOwner): ComposeNode =
            ComposeNode(SimpleScreenNode(id, simpleName, simpleName), parent)
    }
}
