package com.instabug.library.sessionreplay.monitoring

import androidx.annotation.VisibleForTesting
import com.instabug.library.core.eventbus.eventpublisher.Subscriber
import com.instabug.library.screenshot.analytics.AnalyticsEvent
import com.instabug.library.sessionreplay.SRLoggingController.Status
import com.instabug.library.sessionreplay.SR_LOG_TAG
import com.instabug.library.sessionreplay.model.SRLog
import com.instabug.library.sessionreplay.model.SRLogType
import com.instabug.library.util.extenstions.logDWithThreadName
import com.instabug.library.visualusersteps.ReproConfigurationsProvider

interface SRLoggingMonitor : Subscriber<AnalyticsEvent> {
    fun trackLog(log: SRLog, @Status status: Int)
    fun trackScreenshot(@Status status: Int)
    fun analyzeAndTrackException(throwable: Throwable?)
    fun analyzeAndTrackConfigurations(configurations: SRConfigurations)
}

interface ControlledSRLoggingMonitor : SRLoggingMonitor {
    fun init(sessionId: String)
    fun shutdown()
}

class SimpleControlledSRLoggingMonitor(
    private val dataStore: SRMonitoringSpansDataStore,
    private val configurationsProvider: ReproConfigurationsProvider
) : ControlledSRLoggingMonitor {

    @VisibleForTesting
    var analytics: SRAnalytics? = null
    private val trackableStatuses: Int
        /**
         * Given current monitoring specs, only success & limits violation class of [Status]
         * is desired for tracking.
         * Hence, this [Int] returning getter returns a binary representation that includes those
         * two classes.
         * (001-00000) or (100-00000) = (101-00000)
         */
        get() = Status.EligibleForStoring or Status.LimitsViolation

    override fun init(sessionId: String) {
        "[Monitoring] Initializing logging controller".logDWithThreadName(SR_LOG_TAG)
        analytics = SRAnalytics(sessionId)
    }

    override fun trackLog(log: SRLog, @Status outcome: Int) {
        if (!isTrackableOutcome(outcome)) return
        trackLogTypeCounts(log)
        trackDrops(outcome) { sessionStorageViolationDrops++ }
        analytics?.copy()?.also(dataStore::store)
    }

    override fun trackScreenshot(@Status outcome: Int) {
        if (!isTrackableOutcome(outcome)) return
        mutateAnalytics { screenshotsCount++ }
        trackDrops(outcome) { screenshotsStorageViolationDrops++ }
        analytics?.copy()?.also(dataStore::store)
    }

    override fun analyzeAndTrackException(throwable: Throwable?) {
        throwable ?: return
        getDeepestMonitoredException(throwable)
            ?.let { exception -> mutateAnalytics { errors.add(exception.code) } }
        analytics?.copy()?.also(dataStore::store)
    }

    override fun analyzeAndTrackConfigurations(configurations: SRConfigurations) {
        mutateAnalytics { errors.addAll(configurations.generateErrorCodes()) }
        analytics?.copy()?.also(dataStore::store)
    }

    override fun shutdown() {
        "[Monitoring] Shutting down logging controller".logDWithThreadName(SR_LOG_TAG)
        analytics = null
    }

    /**
     * Given the [Status] input, this computation determines whether the it's of a desired class or not
     * Example:
     * EnablementViolation (not desirable) -> (010-xxxxx)
     * Calculated desirable statuses from [trackableStatuses] -> (101-00000)
     * (010-xxxxx) and (101-00000) = (000-00000) which is not > 0 (fail)
     *
     * Any LimitsViolation (desirable) -> (100-xxxxx)
     * Calculated desirable statuses from [trackableStatuses] -> (101-00000)
     * (100-xxxxx) and (101-00000) = (100-00000) (success)
     */
    private fun isTrackableOutcome(@Status outcome: Int) =
        ((outcome and trackableStatuses) > 0)

    private fun trackLogTypeCounts(log: SRLog) {
        mutateAnalytics {
            when (log.logType) {
                SRLogType.IBG_LOG -> ibgLogsCount++
                SRLogType.NETWORK_LOG -> networkLogsCount++
                SRLogType.USER_STEP -> userStepsCount++
                SRLogType.SCREENSHOT -> screenshotsMetadataCount++
            }
        }
    }

    private fun trackDrops(
        @Status status: Int,
        onSessionStorageViolation: SRAnalytics.() -> Unit
    ) {
        if (!isTrackableDropStatus(status)) return
        mutateAnalytics {
            when (status) {
                Status.AggregateStorageLimitViolation -> aggregateStorageViolation = true
                Status.SessionStorageLimitViolation -> onSessionStorageViolation(this)
                Status.SamplingViolation -> samplingDrops++
                else -> Unit
            }
        }
    }

    private fun isTrackableDropStatus(@Status status: Int) =
        (Status.LimitsViolation and status) > 0

    private fun mutateAnalytics(operation: SRAnalytics.() -> Unit) {
        analytics?.apply(operation)
    }

    private fun getDeepestMonitoredException(throwable: Throwable): SRMonitoredException? {
        var result: Throwable = throwable
        var tester = throwable.cause
        while (tester != null) {
            if (tester is SRMonitoredException) result = tester
            tester = tester.cause
        }
        return (result as? SRMonitoredException)
    }

    override fun onNewEvent(event: AnalyticsEvent) {
        if (event is AnalyticsEvent.ErrorEvent)
            analytics
                .takeIf { configurationsProvider.isReproScreenshotsAvailable }
                ?.errors?.add(event.errorCode)
                ?.also { analytics?.copy()?.also(dataStore::store) }
    }
}