package com.instabug.library.networkv2.authorization;

import android.util.Base64;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.library.Constants;
import com.instabug.library.internal.servicelocator.CoreServiceLocator;
import com.instabug.library.networkv2.authorization.ispx.ISP;
import com.instabug.library.networkv2.request.Request;
import com.instabug.library.networkv2.request.RequestMethod;
import com.instabug.library.util.InstabugSDKLogger;

import java.io.IOException;
import java.util.UUID;

import kotlin.random.Random;

public class NetworkOfficer {
    @VisibleForTesting
    public static synchronized String generateNonce() {
        return UUID.randomUUID().toString();
    }

    public static synchronized String getAuthHeader(Request request) throws Exception {
        String nonce = generateNonce();
        long currentTimeMillis = System.currentTimeMillis();
        ISP implementation = CoreServiceLocator.getSigningImplementation();

        StringBuilder authHeaderBuilder = new StringBuilder(implementation.BQfA(Random.Default.nextInt(-1073741824, -1)));
        // adding nonce
        authHeaderBuilder.append(nonce);
        String separator = implementation.Bp1C(Random.Default.nextInt(2080374784, 2147483647));
        authHeaderBuilder.append(separator);
        String clientId = null;
        String signatureHash = null;
        try {
            clientId = implementation.Ulvs(Random.Default.nextInt(1610612736, 2147483647));
            signatureHash = getSignatureHash(implementation.Vx8Q(Random.Default.nextInt(-536870912, -1)),
                    signatureBaseString(request, clientId, nonce, currentTimeMillis));
        } catch (UnsatisfiedLinkError e) {
            // to pass unit tests only.
        }
        // adding signature hash
        if (signatureHash != null) {
            authHeaderBuilder.append(signatureHash);
        }
        authHeaderBuilder.append(separator);
        // adding app token
        if (clientId != null) {
            authHeaderBuilder.append(clientId);
        }
        authHeaderBuilder.append(separator);
        // adding time stamp
        authHeaderBuilder.append(currentTimeMillis);

        return authHeaderBuilder.toString();
    }

    private static synchronized String getSignatureHash(String key, String data) throws Exception {
        return DataManipulationUtility.encodeHMAC(key, data);
    }

    @VisibleForTesting
    @NonNull
    public static synchronized String signatureBaseString(Request request, String appToken, String nonce, long currentTimeMillis) {
        StringBuilder signatureBaseString = new StringBuilder();
        // REQUEST METHOD
        if (request.getRequestMethod() != null) {
            signatureBaseString.append(request.getRequestMethod());
        }
        // ENCODED REQUEST URL
        signatureBaseString.append(DataManipulationUtility.encodeUrl(request.getRequestUrl()));
        // APP TOKEN
        signatureBaseString.append(appToken);
        // NONCE
        signatureBaseString.append(nonce);

        // BODY
        if ((request.getRequestMethod().equals(RequestMethod.POST) ||
                request.getRequestMethod().equals(RequestMethod.PUT)) &&
                request.getRequestBody() != null) {
            String hashBody = hashBody(request);
            if (hashBody != null) {
                if (!hashBody.isEmpty())
                    signatureBaseString.append(hashBody);
            } else {
                InstabugSDKLogger.e(Constants.LOG_TAG,
                        "failed to hash Request body");
                return "";
            }
        }

        // TIMESTAMP
        signatureBaseString.append(currentTimeMillis);

        return signatureBaseString.toString();
    }


    /**
     * A method to hash the body there are two types of body.
     * 1. string body then trigger {@link NetworkOfficer#hashNonMultipartRequests(Request)}
     * 2. multipart body then trigger {@link NetworkOfficer#hashMultipart(Request)}
     *
     * @param request which will hash its body.
     * @return return null if any error occurred, or empty if the body is empty, or hashed value if everything is working fine.
     */
    @Nullable
    @VisibleForTesting
    static synchronized String hashBody(Request request) {
        if (request.isMultiPartRequest()) {
            return "";
        } else {
            return hashNonMultipartRequests(request);
        }
    }

    /**
     * a method to hash multipart
     *
     * @param request which will hash its body.
     * @return empty string if file size is zero, or hashed file, or null if any error occurred.
     */
    @Nullable
    private synchronized static String hashMultipart(Request request) {
        String hashedBodyString = null;
        try {
            hashedBodyString = MultipartRequestOfficer.hashMultipartRequest(request);
        } catch (IOException e) {
            InstabugSDKLogger.e(Constants.LOG_TAG,
                    "get signature base string",
                    e);
        } catch (OutOfMemoryError e) {
            InstabugSDKLogger.e(Constants.LOG_TAG,
                    "OOM: Failed to get signature base string",
                    e);
        }

        return hashedBodyString;

    }

    /**
     * a method to hash request's body that's not multipart.
     *
     * @param request which will hash its body.
     * @return empty string if body is empty or null, or hashed body, or null if any error occurred.
     */
    @Nullable
    private static synchronized String hashNonMultipartRequests(@NonNull Request request) {
        try {
            if (request.getRequestBody() == null || request.getRequestBody().isEmpty()) {
                return "";
            }
            byte[] gZipRequest = DataManipulationUtility.gzip(request.getRequestBody());
            String encodedGzip = Base64.encodeToString(gZipRequest, Base64.NO_WRAP);

            String md5Hash = DataManipulationUtility.md5Hash(encodedGzip);

            if (md5Hash == null || md5Hash.isEmpty())
                return null;
            return md5Hash;

            // GZipping BODY

        } catch (IOException e) {
            InstabugSDKLogger.e(Constants.LOG_TAG,
                    "Failed to get signature base string",
                    e);
        } catch (OutOfMemoryError e) {
            InstabugSDKLogger.e(Constants.LOG_TAG,
                    "OOM: Failed to get signature base string",
                    e);
        }
        return null;
    }

}
