package com.instabug.library.model.v3Session

import com.instabug.library.model.common.Extractable
import com.instabug.library.model.common.Session
import com.instabug.library.model.v3Session.IBGSessionKeys.DURATION_KEY
import com.instabug.library.model.v3Session.IBGSessionKeys.ID_KEY
import com.instabug.library.model.v3Session.IBGSessionKeys.IS_STITCHED_SESSION_KEY
import com.instabug.library.model.v3Session.IBGSessionKeys.RANDOM_PARTIAL_SESSION_ID
import com.instabug.library.model.v3Session.IBGSessionKeys.SESSION_REPLAY_ENABLED
import com.instabug.library.model.v3Session.IBGSessionKeys.SESSION_V2_SENT_KEY
import com.instabug.library.model.v3Session.RatingDialogDetection.REVIEW_PROMPT_KEY
import com.instabug.library.sessionV3.di.IBGSessionServiceLocator
import com.instabug.library.sessionV3.providers.IBGSessionDataProvider
import com.instabug.library.sessionV3.ratingDialogDetection.RatingDialogData
import com.instabug.library.util.TimeUtils
import org.json.JSONObject
import java.util.UUID
import java.util.concurrent.TimeUnit

typealias IBGSessions = List<IBGSession>

class IBGSessionData(val featureKey: String, val featureData: JSONObject)

data class IBGInMemorySession(val startTime: StartTime, val id: String, val randomID: UInt) {
    companion object Factory {
        fun create(
            event: SessionEvent.Start,
            dataProvider: IBGSessionDataProvider = IBGSessionServiceLocator.sessionDataProvider
        ) = IBGInMemorySession(
            startTime = StartTime.create(event),
            id = UUID.randomUUID().toString(),
            dataProvider.randomID
        )
    }
}

sealed class SessionEvent(
    val eventTimeMicro: Long = TimeUtils.currentTimeStampMicroSeconds(),
    val eventNanoTime: Long = TimeUtils.nanoTime()
) {
    class Start(
        val startedManually: Boolean = false,
        val isForeground: Boolean = false
    ) : SessionEvent()

    class Stop : SessionEvent()
    class End(val isCrashed: Boolean = false) : SessionEvent()

    class RatingDialogDataReady(val ratingDialogData: RatingDialogData) : SessionEvent()

    override fun toString(): String = when (this) {
        is End -> "End"
        is Start -> "Start"
        is Stop -> "Stop"
        is RatingDialogDataReady -> "RatingDialogDataReady"
    }
}

sealed class IBGSessionState {
    class Started(val session: Session) : IBGSessionState()
    object Finished : IBGSessionState()
}

data class IBGSession(
    val serial: Long = -1,
    val id: String,
    val randomID: UInt,
    val userData: SessionUserData,
    val appData: SessionAppData,
    val stitchingState: StitchingState,
    val isV2SessionSent: Boolean,
    val startTime: StartTime,
    val productionUsage: SessionProductionUsage?,
    val durationInMicro: Long = 0L,
    val syncStatus: SyncStatus = SyncStatus.RUNNING,
    val srEnabled: Boolean,
    val isSrEvaluated: Boolean,
    val ratingDialogDetection: String? = null
) : Extractable {
    fun update(
        sessionEvent: SessionEvent,
        dataProvider: IBGSessionDataProvider = IBGSessionServiceLocator.sessionDataProvider
    ): IBGSession {
        return copy(
            isV2SessionSent = dataProvider.isV2SessionSent,
            durationInMicro = calculateDuration(sessionEvent),
            userData = SessionUserData.create(dataProvider),
            appData = SessionAppData.create(dataProvider),
            productionUsage = SessionProductionUsage.create(dataProvider),
            syncStatus = SyncStatus.OFFLINE,
            ratingDialogDetection = updateRatingDataWithCustomStoreRate(sessionEvent)
        )
    }

    private fun updateRatingDataWithCustomStoreRate(sessionEvent: SessionEvent) =
        if (sessionEvent is SessionEvent.End && sessionEvent.isCrashed)
            ratingDialogDetection
        else
            addCustomRatingData(sessionEvent, ratingDialogDetection)

    private fun addCustomRatingData(
        sessionEvent: SessionEvent,
        ratingDialogDetection: String?
    ): String? {
        val timestamp = TimeUnit.MICROSECONDS.toMillis(sessionEvent.eventTimeMicro)
        return IBGSessionServiceLocator
            .manualRatingDetector
            .addCurrentModeToRateDetectionDataJson(timestamp, ratingDialogDetection)
    }

    fun update(srEnabled: Boolean): IBGSession {
        return copy(srEnabled = srEnabled)
    }

    fun toForegroundSession(
        startTime: StartTime,
        dataProvider: IBGSessionDataProvider = IBGSessionServiceLocator.sessionDataProvider
    ): IBGSession {
        return copy(
            startTime = startTime,
            srEnabled = IBGSessionServiceLocator.srEnabled,
            stitchingState = dataProvider.getStitchingState(startTime)
        )
    }

    private fun calculateDuration(sessionEvent: SessionEvent) =
        sessionEvent.eventTimeMicro - startTime.value

    override fun extractFields(map: MutableMap<String, Any>): MutableMap<String, Any> =
        map
            .let(userData::extractFields)
            .let(appData::extractFields)
            .let(startTime::extractFields)
            .apply {
                productionUsage?.extractFields(this)
                put(ID_KEY, id)
                put(SESSION_V2_SENT_KEY, isV2SessionSent)
                stitchingState
                    .takeUnless { stitchingState == StitchingState.BACKGROUND_SESSION }
                    ?.let {
                        put(
                            IS_STITCHED_SESSION_KEY,
                            stitchingState == StitchingState.SESSION_LEAD
                        )
                    }
                put(DURATION_KEY, durationInMicro)
                randomID
                    .takeUnless { randomID.toInt() == -1 }
                    ?.let { put(RANDOM_PARTIAL_SESSION_ID, randomID.toLong()) }
                ratingDialogDetection?.let { put(REVIEW_PROMPT_KEY, it) }
                put(SESSION_REPLAY_ENABLED, srEnabled)
            }

    companion object Factory {
        fun create(
            inMemorySession: IBGInMemorySession,
            dataProvider: IBGSessionDataProvider = IBGSessionServiceLocator.sessionDataProvider,
            srEnabled: Boolean = IBGSessionServiceLocator.srEnabled
        ): IBGSession = with(dataProvider) {
            val sessionReplayEnabled = inMemorySession.startTime
                .takeUnless { it.isBackground }
                ?.let { srEnabled } ?: false
            IBGSession(
                id = inMemorySession.id,
                userData = SessionUserData.create(dataProvider),
                appData = SessionAppData.create(dataProvider),
                isV2SessionSent = dataProvider.isV2SessionSent,
                startTime = inMemorySession.startTime,
                stitchingState = getStitchingState(inMemorySession.startTime),
                productionUsage = SessionProductionUsage.create(dataProvider),
                randomID = inMemorySession.randomID,
                srEnabled = sessionReplayEnabled,
                isSrEvaluated = false
            )
        }
    }
}

enum class SyncStatus { RUNNING, OFFLINE, READY_FOR_SYNC, SYNCED }
enum class StitchingState {
    SESSION_LEAD,
    STITCHED,
    BACKGROUND_SESSION
}
