package com.instabug.library.model

import android.content.Context
import com.instabug.library.Feature
import com.instabug.library.IBGFeature
import com.instabug.library.InstabugFeaturesManager
import com.instabug.library.datahub.HubReportModifier
import com.instabug.library.internal.filestore.ActiveCurrentSpanSelector
import com.instabug.library.internal.storage.cache.db.userAttribute.UserAttributesDbHelper

class StateBuilder(context: Context) : State.Builder(context) {
    private var instabugLogsEnabled = true
    private var experimentsEnabled = true
    private var percentage = State.FULL_STATE_PERCENTAGE
    private var hubDataEnabled = true
    private var stateLogsEnabled = true

    fun withInstabugLogs(enabled: Boolean) =
        this.apply { instabugLogsEnabled =  enabled}

    fun withExperiments(enabled: Boolean) =
        this.apply { experimentsEnabled = enabled }

    fun withPercentage(percentage: Float) =
        this.apply { this.percentage = percentage }

    fun withHubData(enabled: Boolean) =
        this.apply { hubDataEnabled = enabled }

    fun withStateLogs(enabled: Boolean) =
        this.apply { stateLogsEnabled = enabled }

    fun build(): State = buildSimplifiedState().also { state ->
        populateStateLogs(state) {
            populateCommonStateItems(state)
            populateStateExperiments(state)
        }
    }

    fun rebuildStateLogs(state: State) =
        populateStateLogs(state){}

    private inline fun populateStateLogs(state: State, callback: ((State) -> Unit)) {
        val reportModifier: HubReportModifier? = prepareHubReportModifier(state)
        callback.invoke(state)
        populateStateLogItems(state)
        reportModifier?.finish()
    }

    private fun populateStateLogItems(state: State) {
        if (stateLogsEnabled) {
            state.setConsoleLog(getConsoleLog(percentage))
                .setUserSteps(getUserSteps(percentage))
                .setUserData(userData)
                .setUserEvents(getUserEvents(percentage))


            if (InstabugFeaturesManager.getInstance().getFeatureState(IBGFeature.SESSION_PROFILER)
                == Feature.State.ENABLED
            ) {
                state.setSessionProfilerTimeline(getSessionProfilerTimeline(percentage))
            }
        }
    }

    private fun populateCommonStateItems(state: State) {
        state.setBuildPercentage(percentage)
            .setUserEmail(userEmail)
            .setUserName(userName)
            .setPushToken(pushNotificationToken)
            .setTags(tags)
            .setUserAttributes(UserAttributesDbHelper.getSDKUserAttributes())
            .setCurrentActivity(getCurrentActivity())
    }

    private fun populateStateExperiments(state: State) {
        if (experimentsEnabled) {
            state.setExperiments(getExperiments(percentage))
        }
    }

    private fun prepareHubReportModifier(state: State): HubReportModifier? =
        takeIf { hubDataEnabled && stateLogsEnabled }
            ?.let {
                HubReportModifier.Builder()
                    .withIBGLogs(instabugLogsEnabled)
                    .buildWithDefaultStores()
            }
            ?.also { it.prepare(state, ActiveCurrentSpanSelector()) }
}