package com.instabug.library.internal.storage;

import android.annotation.SuppressLint;
import android.content.Context;
import android.util.Base64;

import androidx.annotation.NonNull;

import com.instabug.library.Constants;
import com.instabug.library.Instabug;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.memory.MemoryUtils;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.io.UnsupportedEncodingException;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.NoSuchAlgorithmException;

import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;


import static com.instabug.library.util.FileUtils.getPathWithDecryptedFlag;
import static com.instabug.library.util.FileUtils.getPathWithEncryptedFlag;
import static com.instabug.library.util.FileUtils.isReproStepFile;
import static javax.crypto.Cipher.DECRYPT_MODE;
import static javax.crypto.Cipher.ENCRYPT_MODE;

@Deprecated
public class Encryptor {
    private static final String AES = "AES/ECB/NoPadding";
    private static final String AES_CBC_WITH_PADDING = "AES/CBC/PKCS5PADDING";
    public static final String LINE_FEED = "\n\r";
    private static final String CHARSET = "UTF-8";
    private static final int NUMBER_BYTES_TO_PROCESS = 256;

    static {
        try {
            System.loadLibrary("ibg-native");
        } catch (UnsatisfiedLinkError e) {
            // to pass unit tests only.
        }
    }

    public static native String getKey();

    public static native String getCBCSecretKey();

    public static native String getCBCIVParamterKey();

    public static boolean encrypt(@NonNull String path) throws UnsatisfiedLinkError {
        File file = new File(path);
        boolean isFileEncrypted = fileProcessor(ENCRYPT_MODE, file);
        if (isFileEncrypted && isReproStepFile(path)) {
            String newPath = getPathWithEncryptedFlag(path);
            if (!newPath.equals("")) {
                file.renameTo(new File(newPath));
            }
        }
        return isFileEncrypted;
    }

    public static boolean decrypt(@NonNull String filePath) throws UnsatisfiedLinkError {
        File file = new File(filePath);
        boolean isFileDecrypted = fileProcessor(DECRYPT_MODE, file);
        if (isFileDecrypted && isReproStepFile(filePath)) {
            String newPath = getPathWithDecryptedFlag(filePath);
            file.renameTo(new File(newPath));
        }
        return isFileDecrypted;
    }

    public static ProcessedBytes decryptOnTheFly(String filePath) throws UnsatisfiedLinkError {
        File file = new File(filePath);
        return fileDecryptionOnTheFlyProcessor(file);
    }

    @SuppressLint("RESOURCE_LEAK")
    private static boolean fileProcessor(int cipherMode, File file) {
        Key secretKey = null;
        FileInputStream fis = null;
        RandomAccessFile raf = null;
        Context context = Instabug.getApplicationContext();
        if (context != null && !MemoryUtils.isLowMemory(context)) {
            try {
                secretKey = new SecretKeySpec(getKey().getBytes(CHARSET), AES);
                Cipher cipher = Cipher.getInstance(AES);
                cipher.init(cipherMode, secretKey);

                // Get the first NUMBER_BYTES_TO_PROCESS bytes and process them
                fis = new FileInputStream(file);
                byte[] bytesToProcess;
                bytesToProcess = new byte[NUMBER_BYTES_TO_PROCESS];
                fis.read(bytesToProcess, 0, bytesToProcess.length);
                byte[] processedBytes = cipher.doFinal(bytesToProcess);

                // Override the file with the processed bytes
                // Mode "rws" according to docs is used for reading/writing file synchronously
                raf = new RandomAccessFile(file, "rws");
                raf.write(processedBytes, 0, processedBytes.length);

                return true;
            } catch (NoSuchPaddingException | NoSuchAlgorithmException |
                    InvalidKeyException | IOException | BadPaddingException
                    | IllegalBlockSizeException | OutOfMemoryError | UnsatisfiedLinkError e) {
                if (secretKey != null) {
                    resetCipher(cipherMode, secretKey);
                }
                String cipherModeDescription = "";
                if (cipherMode == ENCRYPT_MODE) {
                    cipherModeDescription = "encrypting";
                } else if (cipherMode == DECRYPT_MODE) {
                    cipherModeDescription = "decrypting";
                }
                InstabugSDKLogger.e(Constants.LOG_TAG,
                        String.format("Error: %s occurred while %s file in path: %s", e, cipherModeDescription, file.getPath()));
            } finally {
                try {
                    if (fis != null) {
                        fis.close();
                    }
                    if (raf != null) {
                        raf.close();
                    }
                } catch (IOException e) {
                }
            }
        }
        return false;
    }

    @SuppressLint("RESOURCE_LEAK")
    private static ProcessedBytes fileDecryptionOnTheFlyProcessor(File file) {
        Key secretKey = null;
        FileInputStream fis = null;
        RandomAccessFile raf = null;
        try {
            secretKey = new SecretKeySpec(getKey().getBytes(CHARSET), AES);
            Cipher cipher = Cipher.getInstance(AES);
            cipher.init(DECRYPT_MODE, secretKey);

            // Get the first NUMBER_BYTES_TO_PROCESS bytes and decrypt them
            fis = new FileInputStream(file);
            byte[] bytesToEncrypt = new byte[NUMBER_BYTES_TO_PROCESS];
            fis.read(bytesToEncrypt, 0, bytesToEncrypt.length);
            byte[] encryptedBytes = cipher.doFinal(bytesToEncrypt);
            fis.close();

            // Override the file with the decrypted bytes
            // Mode "rws" according to docs is used for reading/writing file synchronously
            raf = new RandomAccessFile(file, "rws");
            raf.write(encryptedBytes, 0, encryptedBytes.length);

            // Read the whole file
            byte[] decryptedFileBytes = new byte[(int) file.length()];
            read(file, decryptedFileBytes);

            // Rename the file to the unencrypted
            if (isReproStepFile(file.getPath())) {
                String newPath = getPathWithDecryptedFlag(file.getPath());
                file.renameTo(new File(newPath));
            }

            return new ProcessedBytes(decryptedFileBytes, true);
        } catch (NoSuchPaddingException | NoSuchAlgorithmException |
                InvalidKeyException | IOException | BadPaddingException | IllegalBlockSizeException | OutOfMemoryError e) {
            if (secretKey != null) {
                resetCipher(DECRYPT_MODE, secretKey);
            }
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error: " + e + " occurred while decrypting path: " + file.getPath());
        } finally {
            try {
                if (fis != null) {
                    fis.close();
                }
                if (raf != null) {
                    raf.close();
                }
            } catch (IOException e) {
            }
        }
        return new ProcessedBytes(new byte[0], false);
    }

    public static void read(File inputFile, byte[] outputBytes) throws IOException {
        FileInputStream fis = null;
        try {
            fis = new FileInputStream(inputFile);
            fis.read(outputBytes);
        } finally {
            if (fis != null) {
                fis.close();
            }
        }
    }

    private static void resetCipher(int cipherMode, Key secretKey) {
        if (secretKey != null) {
            try {
                Cipher.getInstance(AES).init(cipherMode, secretKey);
            } catch (NoSuchPaddingException | NoSuchAlgorithmException
                    | InvalidKeyException ex) {
                InstabugSDKLogger.e(Constants.LOG_TAG, String.format("Error: %s occurred while resetting the Cipher instance.", ex));
            }
        }
    }

    /**
     * A method to encrypt msg using  {@link Encryptor#AES_CBC_WITH_PADDING} algorithm
     *
     * @param msg to be encrypted.
     * @return
     * @throws NoSuchPaddingException
     * @throws NoSuchAlgorithmException
     * @throws InvalidKeyException
     * @throws UnsupportedEncodingException
     * @throws BadPaddingException
     */
    public static byte[] encryptMsgs(String msg) throws NoSuchPaddingException
            , NoSuchAlgorithmException, InvalidKeyException, UnsupportedEncodingException,
            BadPaddingException, InvalidAlgorithmParameterException, IllegalBlockSizeException {
        IvParameterSpec iv;
        SecretKeySpec skeySpec;
        try {
            iv = new IvParameterSpec(getCBCIVParamterKey().getBytes(CHARSET));
            skeySpec = new SecretKeySpec(getCBCSecretKey().getBytes(CHARSET), AES);
        } catch (UnsatisfiedLinkError e) {
            return Base64.encodeToString(msg.getBytes(CHARSET), Base64.NO_WRAP).getBytes(CHARSET);
        }
        Cipher cipher = Cipher.getInstance(AES_CBC_WITH_PADDING);
        cipher.init(ENCRYPT_MODE, skeySpec, iv);

        byte[] encrypted = cipher.doFinal(msg.getBytes(CHARSET));
        return Base64.encodeToString(encrypted, Base64.NO_WRAP).getBytes(CHARSET);

    }
}
