package com.instabug.library.instacapture.screenshot;

import static com.instabug.library.util.DisplayUtils.getDisplayHeightInPx;
import static com.instabug.library.util.DisplayUtils.getDisplayWidthInPx;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Paint;
import android.graphics.PorterDuff;
import android.graphics.PorterDuffXfermode;
import android.graphics.Rect;
import android.opengl.GLES20;
import android.opengl.GLSurfaceView;
import android.os.Build;
import android.util.Pair;
import android.view.TextureView;
import android.view.View;
import android.view.ViewGroup;
import android.view.WindowManager;
import android.webkit.WebView;
import androidx.annotation.IdRes;
import androidx.annotation.Nullable;
import com.instabug.library.Constants;
import com.instabug.library.Instabug;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.instacapture.exception.ScreenCapturingFailedException;
import com.instabug.library.instacapture.screenshot.pixelcopy.DialogUiRenderer;
import com.instabug.library.instacapture.screenshot.pixelcopy.PixelCopyDelegate;
import com.instabug.library.instacapture.utility.Memory;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.ActivitySecureFlagDetector;
import com.instabug.library.util.InstabugSDKLogger;
import java.nio.IntBuffer;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CountDownLatch;
import javax.microedition.khronos.egl.EGL10;
import javax.microedition.khronos.egl.EGLContext;
import javax.microedition.khronos.opengles.GL10;
import io.reactivexport.Observable;
import io.reactivexport.ObservableOnSubscribe;
import io.reactivexport.ObservableSource;
import io.reactivexport.android.schedulers.AndroidSchedulers;
import io.reactivexport.functions.Function;
import io.reactivexport.schedulers.Schedulers;

/**
 * Created by tarek on 5/17/16.
 */
public final class ScreenshotTaker {


    private ScreenshotTaker() {
    }

    /**
     * Capture screenshot for the current activity and return bitmap of it.
     *
     * @param activity        current activity.
     * @param ignoredViewsIds from the screenshot.
     * @return Bitmap of screenshot.
     * @throws ScreenCapturingFailedException if unexpected error is occurred during capturing
     *                                        screenshot
     */
    public static Observable<Bitmap> getScreenshotBitmap(Activity activity, @Nullable @IdRes int[] ignoredViewsIds) {
        if (activity == null) {
            throw new IllegalArgumentException("Parameter activity cannot be null.");
        }

        boolean isFlagSecureEnabled = ActivitySecureFlagDetector.isFlagSecure(activity);
        if (isFlagSecureEnabled && !SettingsManager.getInstance().shouldIgnoreFlagSecure()) {
            throw new ScreenCapturingFailedException("FLAG_SECURE is enabled for activity "
                    + activity.getClass().getName() + " . Not capturing screenshot.");
        }

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            return PixelCopyDelegate.takeScreenshot(activity, ignoredViewsIds);
        } else {

            View main = activity.getWindow().getDecorView();

            final List<RootViewInfo> viewRoots = FieldHelper.getRootViews(activity, ignoredViewsIds);
            final Bitmap bitmap;
            try {
                if (main.getWidth() * main.getHeight() * 4 < Memory.getFreeMemory(activity)) {
                    // ARGB_8888 store each pixel in 4 bytes
                    bitmap = Bitmap.createBitmap(main.getWidth(), main.getHeight(), Bitmap.Config.ARGB_8888);
                } else {
                    // RGB_565 store each pixel in 2 bytes
                    bitmap = Bitmap.createBitmap(main.getWidth(), main.getHeight(), Bitmap.Config.RGB_565);
                }

            } catch (IllegalArgumentException | OutOfMemoryError e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, e.getMessage() == null ? "error while capturing screenshot" : e.getMessage(), e);
                return Observable.error(e);
            }

            return drawRootsToBitmap(viewRoots, bitmap, ignoredViewsIds, activity);
        }
    }

    private static Observable<Bitmap> drawRootsToBitmap(final List<RootViewInfo> viewRoots, final Bitmap bitmap,
                                                        @Nullable @IdRes final int[] ignoredViewsIds, final Activity activity) {
        if (viewRoots != null && viewRoots.size() > 1) {
            while (viewRoots.size() > 1) {
                viewRoots.remove(viewRoots.size() - 1);
            }
        }
        return Observable.fromIterable(viewRoots)
                .flatMap((Function<RootViewInfo, ObservableSource<Bitmap>>) rootViewInfo -> drawRootToBitmap(rootViewInfo, bitmap, ignoredViewsIds, activity));
    }

    private static Observable<Bitmap> drawRootToBitmap(final RootViewInfo rootViewInfo, final Bitmap bitmap,
                                                       @Nullable @IdRes final int[] ignoredViewsIds, final Activity activity) {
        return Observable.create((ObservableOnSubscribe<Pair<Canvas, HashMap<View, Integer>>>) emitter -> {
                    // support dim screen
                    if ((rootViewInfo.getLayoutParams().flags & WindowManager.LayoutParams.FLAG_DIM_BEHIND)
                            == WindowManager.LayoutParams.FLAG_DIM_BEHIND) {
                        Canvas dimCanvas = new Canvas(bitmap);

                        int alpha = (int) (255 * rootViewInfo.getLayoutParams().dimAmount);
                        dimCanvas.drawARGB(alpha, 0, 0, 0);
                    }

                    Canvas canvas = new Canvas(bitmap);
                    canvas.translate(rootViewInfo.getLeft(), rootViewInfo.getTop());

                    HashMap<View, Integer> ignoredViews = new HashMap<>();
                    if (ignoredViewsIds != null) {
                        for (int i = 0; i < ignoredViewsIds.length; i++) {
                            final View ignoredView = rootViewInfo.getView().findViewById(ignoredViewsIds[i]);
                            if (ignoredView != null) {
                                ignoredViews.put(ignoredView, ignoredView.getVisibility());
                            }
                        }
                    }
                    emitter.onNext(new Pair<>(canvas, ignoredViews));
                }).observeOn(AndroidSchedulers.mainThread())
                .map(pair -> {
                    for (View view : pair.second.keySet()) {
                        view.setVisibility(View.INVISIBLE);
                    }

                    //draw views
                    rootViewInfo.getView().draw(pair.first);
                    return pair;
                }).observeOn(Schedulers.io())
                .map(pair -> {
                    //Draw undrawable views
                    drawUnDrawableViews(rootViewInfo.getView(), pair.first);
                    if (SettingsManager.getInstance().getShouldCaptureDialog()){
                        DialogUiRenderer.tryRenderDialogs(activity, bitmap);
                    }
                    return pair.second;
                }).observeOn(AndroidSchedulers.mainThread())
                .map(new Function<HashMap<View, Integer>, Bitmap>() {
                    @Override
                    public Bitmap apply(HashMap<View, Integer> viewIntegerHashMap) throws Exception {
                        for (Map.Entry<View, Integer> entry : viewIntegerHashMap.entrySet()) {
                            entry.getKey().setVisibility(entry.getValue());
                        }
                        return bitmap;
                    }
                })
                .subscribeOn(Schedulers.io());
    }

    public static boolean isVisible(final View view) {
        if (view.getVisibility() != View.VISIBLE) {
            return false;
        }
        final Rect actualPosition = new Rect();
        view.getGlobalVisibleRect(actualPosition);
        Context context = Instabug.getApplicationContext();
        int displayWidthInPx = context == null ? 0 : getDisplayWidthInPx(context);
        int displayHeightInPx = context == null ? 0 : getDisplayHeightInPx(context);
        final Rect screen = new Rect(0, 0, displayWidthInPx, displayHeightInPx);
        return actualPosition.intersect(screen);
    }

    public static Rect getVisibleRect(@Nullable final View childView) {
        Rect clippedRect = new Rect();
        if (childView == null || childView.getVisibility() != View.VISIBLE || childView.getRootView().getParent() == null) {
            return new Rect(0, 0, 0, 0);
        }

        if (!childView.getGlobalVisibleRect(clippedRect)) {
            // Not visible
            return new Rect(0, 0, 0, 0);
        }
        // to get view areas
//        final long visibleViewArea = (long) clippedRect.height() * clippedRect.width();
//        final long totalViewArea = (long) childView.getHeight() * childView.getWidth();

        return clippedRect;
    }

    public static Rect getVisibleRect(@Nullable final View childView, Rect rect) {
        Rect clippedRect = new Rect();
        if (childView == null || childView.getVisibility() != View.VISIBLE || childView.getRootView().getParent() == null) {
            return new Rect(0, 0, 0, 0);
        }

        if (!childView.getGlobalVisibleRect(clippedRect)) {
            // Not visible
            return new Rect(0, 0, 0, 0);
        }
        // to get view areas
//        final long visibleViewArea = (long) clippedRect.height() * clippedRect.width();
//        final long totalViewArea = (long) childView.getHeight() * childView.getWidth();

        return clippedRect;
    }

    private static ArrayList<View> drawUnDrawableViews(View v, Canvas canvas) {

        if (!(v instanceof ViewGroup)) {
            ArrayList<View> viewArrayList = new ArrayList<>();
            viewArrayList.add(v);
            return viewArrayList;
        }

        ArrayList<View> result = new ArrayList<>();

        ViewGroup viewGroup = (ViewGroup) v;
        for (int i = 0; i < viewGroup.getChildCount(); i++) {

            View child = viewGroup.getChildAt(i);

            ArrayList<View> viewArrayList = new ArrayList<>();
            viewArrayList.add(v);
            viewArrayList.addAll(drawUnDrawableViews(child, canvas));

            int[] childLocationOnScreen = new int[2];
            child.getLocationOnScreen(childLocationOnScreen);

            if (child instanceof TextureView) {
                drawTextureView((TextureView) child, childLocationOnScreen, canvas);
            }

            if (child instanceof GLSurfaceView) {
                drawGLSurfaceView((GLSurfaceView) child, childLocationOnScreen, canvas);
            }

            if (child instanceof WebView) {
                drawWebView((WebView) child, canvas);
            }

            result.addAll(viewArrayList);
        }
        return result;
    }

    public static void drawGLSurfaceView(GLSurfaceView surfaceView, int[] locationOnScreen,
                                         Canvas canvas) {
        InstabugSDKLogger.v(Constants.LOG_TAG, "Drawing GLSurfaceView");

        if (surfaceView.getWindowToken() != null) {

            final int width = surfaceView.getWidth();
            final int height = surfaceView.getHeight();

            final int x = 0;
            final int y = 0;
            int[] b = new int[width * (y + height)];

            final IntBuffer ib = IntBuffer.wrap(b);
            ib.position(0);

            //To wait for the async call to finish before going forward
            final CountDownLatch countDownLatch = new CountDownLatch(1);
            surfaceView.queueEvent(() -> {
                EGL10 egl = (EGL10) EGLContext.getEGL();
                egl.eglWaitGL();
                GL10 gl = (GL10) egl.eglGetCurrentContext().getGL();

                gl.glFinish();

                try {
                    Thread.sleep(200);
                } catch (InterruptedException e) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong" + e);
                }

                gl.glReadPixels(x, 0, width, y + height, GLES20.GL_RGBA, GLES20.GL_UNSIGNED_BYTE, ib);
                countDownLatch.countDown();
            });

            try {
                countDownLatch.await();
            } catch (InterruptedException e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong" + e);
            }
            int[] bt = new int[width * height];
            int i = 0;
            for (int k = 0; i < height; k++) {
                for (int j = 0; j < width; j++) {
                    int pix = b[(i * width + j)];
                    int pb = pix >> 16 & 0xFF;
                    int pr = pix << 16 & 0xFF0000;
                    int pix1 = pix & 0xFF00FF00 | pr | pb;
                    bt[((height - k - 1) * width + j)] = pix1;
                }
                i++;
            }

            Bitmap sb = Bitmap.createBitmap(bt, width, height, Bitmap.Config.ARGB_8888);
            Paint paint = new Paint();
            paint.setXfermode(new PorterDuffXfermode(PorterDuff.Mode.DST_ATOP));
            canvas.drawBitmap(sb, locationOnScreen[0], locationOnScreen[1], paint);
            sb.recycle();
        }
    }


    public static void drawTextureView(TextureView textureView, int[] locationOnScreen,
                                       Canvas canvas) {
        InstabugSDKLogger.v(Constants.LOG_TAG, "Drawing TextureView");

        try {
            Bitmap textureViewBitmap = textureView.getBitmap();
            if (textureViewBitmap != null) {
                Paint paint = new Paint();
                paint.setXfermode(new PorterDuffXfermode(PorterDuff.Mode.DST_ATOP));
                canvas.drawBitmap(textureViewBitmap, locationOnScreen[0], locationOnScreen[1], paint);
                textureViewBitmap.recycle();
            }
        } catch (OutOfMemoryError outOfMemoryError) {
            InstabugCore.reportError(outOfMemoryError, "Drawing textureView failed due to an OOM: " + outOfMemoryError.getMessage());
            InstabugSDKLogger.e(Constants.LOG_TAG, "OOM while taking screenshot", outOfMemoryError);
        }
    }


    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public static void drawWebView(WebView webView, Canvas canvas) {

        int layerType = webView.getLayerType();
        if (layerType == View.LAYER_TYPE_HARDWARE) {

            // Set the layer type to LAYER_TYPE_NONE to draw correctly
            webView.setLayerType(View.LAYER_TYPE_NONE, null);

            webView.setDrawingCacheEnabled(true);
            webView.buildDrawingCache(true);
            Bitmap webViewBitmap = webView.getDrawingCache();

            if (webViewBitmap != null) {
                Paint paint = new Paint();
                paint.setXfermode(new PorterDuffXfermode(PorterDuff.Mode.DST_ATOP));

                int[] locationOnScreen = new int[2];
                webView.getLocationOnScreen(locationOnScreen);

                canvas.drawBitmap(webViewBitmap, locationOnScreen[0], locationOnScreen[1], paint);
                webViewBitmap.recycle();
            }

            webView.setDrawingCacheEnabled(false);

            // return layerType
            webView.setLayerType(layerType, null);
        }
    }
}

