package com.instabug.library.apichecker;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.Constants;
import com.instabug.library.Instabug;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.threading.PoolProvider;

public class APIChecker {
    public static final String NOT_BUILT_ERROR_MESSAGE =
            "Instabug API {%s} was called before the SDK is built. To build it, please call Instabug.Builder().build().";
    public static final String MAIN_THREAD_ERROR_MESSAGE =
            "Threading violation: {%s} should only be called from a background thread, but was called from main thread.";
    public static final String NOT_ENABLED_ERROR_MESSAGE =
            "Instabug API {%s} was called while the SDK is disabled. To enable it, please call Instabug.enable().";
    public static final String NOT_EXECUTED_ERROR_MESSAGE =
            "Instabug failed to execute {%s}";

    /**
     * Executes a checks on Instabug state, run a returnable runnable and then return the value
     *
     * @param runnable      the block of code expected to return a value
     * @param fallbackValue the value to be returned in case the execution has experienced an error condition
     * @return the API-specified return value if the execution was successful or the passed fallback value otherwise
     */
    @Nullable
    public static <T> T checkAndGet(@NonNull final String apiName, @NonNull final ReturnableRunnable<T> runnable, @Nullable final T fallbackValue) {
        checkMainThreadCalls(apiName);
        return PoolProvider.getApiExecutor().executeAndGet(() -> {
            try {
                checkBuilt();
                checkEnabled();
                return runnable.run();
            } catch (InstabugNotBuiltStateException exception) {
                logSdkNotBuilt(apiName);
            } catch (InstabugNotEnabledStateException exception) {
                logSdkNotEnabled(apiName);
            } catch (Exception e) {
                logExecutionException(apiName, e);
            }
            return fallbackValue;
        });
    }

    private static void checkMainThreadCalls(String apiName) {
        if (Thread.currentThread().getName().equals("main")) {
            logMainThreadWarning(apiName);
        }
    }

    /**
     * Executes a checks on Instabug state and then run a void runnable
     *
     * @param runnable the block of code expected to return a value
     */
    public static void checkAndRunInExecutor(@NonNull final String apiName, @NonNull final VoidRunnable runnable) {
        PoolProvider.getApiExecutor().execute(() -> {
            try {
                checkBuilt();
                checkEnabled();
                runnable.run();
            } catch (InstabugNotBuiltStateException exception) {
                logSdkNotBuilt(apiName);
            } catch (InstabugNotEnabledStateException exception) {
                logSdkNotEnabled(apiName);
            } catch (Exception e) {
                logExecutionException(apiName, e);
            }
        });
    }


    /**
     * Executes a checks on Instabug state and then run a void runnable
     *
     * @param runnable the block of code expected to return a value and running on Main Thread
     */
    public static void checkAndRunInExecutorThenPostOnMain(@NonNull final String apiName, @NonNull final VoidRunnable runnable) {
        PoolProvider.getApiExecutor().execute(() -> {
            try {
                checkBuilt();
                checkEnabled();
                PoolProvider.postMainThreadTask(() -> {
                    try {
                        runnable.run();
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                });
            } catch (InstabugNotBuiltStateException exception) {
                logSdkNotBuilt(apiName);
            } catch (InstabugNotEnabledStateException exception) {
                logSdkNotEnabled(apiName);
            } catch (Exception e) {
                logExecutionException(apiName, e);
            }
        });
    }

    public static void checkAndRunOrThrow(@NonNull final String apiName, @NonNull final VoidRunnable runnable) throws Exception {
        try {
            checkBuilt();
            checkEnabled();
            PoolProvider.getApiExecutor().execute(() -> {
                try {
                    runnable.run();
                } catch (Exception exception) {
                    logExecutionException(apiName, exception);
                }
            });
        } catch (InstabugNotBuiltStateException exception) {
            logSdkNotBuilt(apiName);
            throw exception;
        } catch (InstabugNotEnabledStateException exception) {
            logSdkNotEnabled(apiName);
            throw exception;
        } catch (Exception exception) {
            logExecutionException(apiName, exception);
            throw exception;

        }
    }

    public static void checkAndRun(@NonNull final String apiName, @NonNull final VoidRunnable runnable) {
        try {
            checkBuilt();
            checkEnabled();
            runnable.run();
        } catch (InstabugNotBuiltStateException exception) {
            logSdkNotBuilt(apiName);
        } catch (InstabugNotEnabledStateException exception) {
            logSdkNotEnabled(apiName);
        } catch (Exception e) {
            logExecutionException(apiName, e);
        }
    }


    /**
     * Executes a checks on Instabug build state only and then run a void runnable
     *
     * @param runnable the block of code expected to return a value
     */
    public static void checkBuilt(@NonNull final String apiName, @NonNull final VoidRunnable runnable) {
        try {
            checkBuilt();
            runnable.run();
        } catch (InstabugNotBuiltStateException exception) {
            APIChecker.logSdkNotBuilt(apiName);
        } catch (Exception e) {
            logExecutionException(apiName, e);
        }
    }

    /**
     * Executes a checks on Instabug enable state only and then run a void runnable
     *
     * @param runnable the block of code expected to return a value
     */
    public static void checkEnable(@NonNull String apiName, @NonNull VoidRunnable runnable) {
        try {
            checkEnabled();
            runnable.run();
        } catch (InstabugNotEnabledStateException exception) {
            APIChecker.logSdkNotEnabled(apiName);
        } catch (Exception e) {
            logExecutionException(apiName, e);
        }
    }

    private static void checkBuilt() throws InstabugNotBuiltStateException {
        if (!Instabug.isBuilt()) {
            throw new InstabugNotBuiltStateException("Instabug API called before Instabug.Builder().build() was called");
        }
    }

    private static void checkEnabled() throws InstabugNotEnabledStateException {
        if (!Instabug.isEnabled()) {
            throw new InstabugNotEnabledStateException("Instabug API called while Instabug SDK was disabled");
        }
    }

    private static void logSdkNotEnabled(String apiName) {
        InstabugSDKLogger.e(Constants.LOG_TAG, String.format(NOT_ENABLED_ERROR_MESSAGE, apiName));
    }

    private static void logSdkNotBuilt(String apiName) {
        InstabugSDKLogger.e(Constants.LOG_TAG, String.format(NOT_BUILT_ERROR_MESSAGE, apiName));
    }

    private static void logMainThreadWarning(String apiName) {
        InstabugSDKLogger.w(Constants.LOG_TAG, String.format(MAIN_THREAD_ERROR_MESSAGE, apiName));
    }

    private static void logExecutionException(String apiName, Exception e) {
        InstabugSDKLogger.e(Constants.LOG_TAG, String.format(NOT_EXECUTED_ERROR_MESSAGE, apiName) + " due to" + e.getMessage());
    }
}
