package com.instabug.library.util.threading

import java.util.LinkedList
import java.util.Queue
import java.util.concurrent.Callable
import java.util.concurrent.ExecutorService
import java.util.concurrent.Future
import java.util.concurrent.FutureTask

interface OrderedExecutorService : ExecutorService {
    fun execute(key: String, runnable: Runnable)
    fun <V> submit(key: String, callable: Callable<V>): Future<V>
}

private const val LOCK = "OrderedExecutor"

class BasicOrderedExecutorService(
    private val delegate: ExecutorService
) : OrderedExecutorService, ExecutorService by delegate {

    @Volatile
    var tasksMap = LinkedHashMap<String, LinkedList<Runnable>>()

    override fun execute(key: String, runnable: Runnable) {
        executeOrdered(key, runnable)
    }

    override fun <V> submit(key: String, callable: Callable<V>): Future<V> {
        val futureTask = FutureTask(callable)
        executeOrdered(key, futureTask)
        return futureTask
    }

    private fun executeOrdered(key: String, runnable: Runnable) {
        var isFirst = false
        var runnableWrapper: RunnableWrapper?
        var tasksQueue: LinkedList<Runnable>?
        synchronized(LOCK) {
            tasksQueue = tasksMap[key]
            takeIf { tasksQueue == null }?.let {
                tasksMap[key] = LinkedList()
                tasksQueue = tasksMap[key]
                isFirst = true
            }

            runnableWrapper = RunnableWrapper(key, runnable, tasksQueue)

            if (!isFirst) {
                tasksQueue?.run {
                    //Second check for safety reasons -> check the related commit message.
                    if (tasksMap[key] == null) {
                        tasksMap[key] = this
                        execute(runnableWrapper)
                    } else {
                        offerLast(runnableWrapper)
                    }
                }
            }
        }

        if (isFirst) {
            execute(runnableWrapper)
        }
    }

    inner class RunnableWrapper(
        private val key: String,
        private val runnable: Runnable,
        private val tasksQueue: Queue<Runnable>?
    ) : Runnable {
        override fun run() {
            try {
                runnable.run()
            } finally {
                var nextTask: Runnable? = null
                synchronized(LOCK) {
                    if (tasksQueue?.isEmpty() == true) {
                        tasksMap.remove(key)
                    } else {
                        nextTask = tasksQueue?.poll()
                    }
                    nextTask?.let {
                        delegate.execute(it)
                    }
                }
            }
        }
    }
}