package com.instabug.library.sessionreplay.monitoring

import android.content.Context
import androidx.annotation.VisibleForTesting
import androidx.annotation.WorkerThread
import com.instabug.library.internal.filestore.AllSpansSelector
import com.instabug.library.internal.filestore.CreateNewFile
import com.instabug.library.internal.filestore.CurrentSpanSelector
import com.instabug.library.internal.filestore.DataAggregator
import com.instabug.library.internal.filestore.DeleteDirectory
import com.instabug.library.internal.filestore.Directory
import com.instabug.library.internal.filestore.FileSelector
import com.instabug.library.internal.filestore.MatchingIDSpanSelector
import com.instabug.library.internal.filestore.MultiSpanOpsFileDataStore
import com.instabug.library.internal.filestore.MultiSpanSelector
import com.instabug.library.internal.filestore.OldSpansSelector
import com.instabug.library.internal.filestore.OperationScopeBuilder
import com.instabug.library.internal.filestore.OverwriteFile
import com.instabug.library.internal.filestore.ReadJSONFromFile
import com.instabug.library.internal.filestore.SpanSelector
import com.instabug.library.internal.filestore.SpansDirectory
import com.instabug.library.internal.filestore.SpansDirectoryFactory
import com.instabug.library.internal.filestore.SpansFileDataStoreLifecycle
import com.instabug.library.sessionreplay.SRExecutionQueues
import com.instabug.library.sessionreplay.SR_LOG_TAG
import com.instabug.library.util.Job
import com.instabug.library.util.JobThrottler
import com.instabug.library.util.extenstions.getOrLogAndReport
import com.instabug.library.util.extenstions.logDWithThreadName
import com.instabug.library.util.extenstions.prefixedWithT
import com.instabug.library.util.extenstions.toDirectory
import com.instabug.library.util.threading.OrderedExecutorService
import java.io.File
import java.util.concurrent.Future

class SRMonitoringDirectory(
    val currentSessionId: String?,
    parent: Directory
) : SpansDirectory<SRMonitoringSessionDirectory>(parent, "sr-monitoring") {
    override val currentDirectory: SRMonitoringSessionDirectory?
        get() = currentSessionId?.let { SRMonitoringSessionDirectory(it, this) }
    override val oldDirectories: List<SRMonitoringSessionDirectory>
        @WorkerThread get() = runCatching {
            listFiles { file -> file.isDirectory && file.name != currentSessionId }
                ?.map { file -> SRMonitoringSessionDirectory(file.name, this) }.orEmpty()
        }.getOrLogAndReport(
            emptyList(),
            "[Monitoring] Error retrieving monitoring old spans directories".prefixedWithT(),
            tag = SR_LOG_TAG
        )

    class Factory(
        private val ctxGetter: () -> Context?,
        private val rootDirGetter: (Context) -> File?
    ) : SpansDirectoryFactory<SRMonitoringDirectory> {
        private var currentSessionId: String? = null

        override fun setCurrentSpanId(spanId: String?) {
            currentSessionId = spanId
        }

        override fun invoke(): SRMonitoringDirectory? =
            ctxGetter()?.let(rootDirGetter)
                ?.toDirectory()
                ?.let { rootDir -> SRMonitoringDirectory(currentSessionId, rootDir) }
    }
}

class SRMonitoringSessionDirectory(
    val sessionId: String,
    parent: SRMonitoringDirectory
) : Directory(parent, sessionId) {
    val analyticsFile: File
        get() = File(this, "analytics.txt")
}

class SRAnalyticsFileSelector : FileSelector<SRMonitoringSessionDirectory> {
    override fun invoke(input: SRMonitoringSessionDirectory): File {
        return input.analyticsFile
    }
}

interface SRMonitoringSpansDataStore :
    MultiSpanOpsFileDataStore<SRAnalytics, SRMonitoringDirectory, SRMonitoringSessionDirectory>,
    SpansFileDataStoreLifecycle<SRMonitoringDirectory> {
    fun storeImmediately(data: SRAnalytics)
}

@VisibleForTesting
const val DEFAULT_THROTTLING_PERIOD = 3_000L

class ThrottledSRMonitoringSpansDataStore(
    private val executor: OrderedExecutorService,
    private val throttler: JobThrottler<SRAnalytics>,
    private val opsDirectoryFactory: SRMonitoringDirectory.Factory
) : SRMonitoringSpansDataStore {

    private var operationsDirectory: SRMonitoringDirectory? = null

    private val storingJob = Job<SRAnalytics> { data ->
        executor.execute(SRExecutionQueues.MonitoringStore) {
            "[Monitoring] Flushing monitoring data to data store".logDWithThreadName(tag = SR_LOG_TAG)
            OperationScopeBuilder(OverwriteFile(SRAnalyticsFileSelector(), data))
                .onSpan(MatchingIDSpanSelector(data.sessionId))
                .buildAndExecute(operationsDirectory)
        }
    }

    override fun init(operationsDirectory: SRMonitoringDirectory): Future<Boolean> =
        executor.submit(SRExecutionQueues.MonitoringStore) {
            "[Monitoring] Initializing data store".logDWithThreadName(tag = SR_LOG_TAG)
            this.operationsDirectory = operationsDirectory; true
        }

    override fun onSpanStarted(spanId: String) {
        executor.execute(SRExecutionQueues.MonitoringStore) {
            "[Monitoring] Data store is starting a new span $spanId.".logDWithThreadName(tag = SR_LOG_TAG)
            operationsDirectory = opsDirectoryFactory.apply { setCurrentSpanId(spanId) }.invoke()
            OperationScopeBuilder(CreateNewFile(SRAnalyticsFileSelector()))
                .onSpan(CurrentSpanSelector())
                .buildAndExecute(operationsDirectory)
        }
    }

    override fun store(log: SRAnalytics) {
        executor.execute(SRExecutionQueues.MonitoringStore) {
            throttler.invoke(storingJob, log, DEFAULT_THROTTLING_PERIOD)
        }
    }

    override fun <Out> retrieve(
        aggregator: DataAggregator<Out>,
        spanSelector: SpanSelector<SRMonitoringDirectory, SRMonitoringSessionDirectory>
    ): Future<Out?> = executor.submit(SRExecutionQueues.MonitoringStore) {
        "[Monitoring] Retrieving single span data from data store".logDWithThreadName(SR_LOG_TAG)
        OperationScopeBuilder(ReadJSONFromFile(SRAnalyticsFileSelector(), aggregator))
            .onSpan(spanSelector)
            .buildAndExecute(operationsDirectory)
    }

    override fun storeImmediately(data: SRAnalytics) {
        executor.execute(SRExecutionQueues.MonitoringStore) {
            storingJob.invoke(data)
        }
    }

    override fun <Out> retrieve(
        aggregator: DataAggregator<Out>,
        spansSelector: MultiSpanSelector<SRMonitoringDirectory, SRMonitoringSessionDirectory>
    ): Future<List<Out?>> = executor.submit(SRExecutionQueues.MonitoringStore) {
        "[Monitoring] Retrieving multi spans data from data store".logDWithThreadName(SR_LOG_TAG)
        OperationScopeBuilder(ReadJSONFromFile(SRAnalyticsFileSelector(), aggregator))
            .onMultiSpans(spansSelector)
            .buildAndExecute(operationsDirectory).orEmpty()
    }

    override fun clear() {
        // Monitoring current span directory is not meant to be cleared.
    }

    override fun delete() {
        // Monitoring current span directory is not meant to be deleted.
        // Multi-span deletion is used instead
    }

    override fun delete(spansSelector: MultiSpanSelector<SRMonitoringDirectory, SRMonitoringSessionDirectory>) {
        executor.execute(SRExecutionQueues.MonitoringStore) {
            "[Monitoring] Deleting multi spans data from data store".logDWithThreadName(SR_LOG_TAG)
            OperationScopeBuilder(DeleteDirectory<SRMonitoringSessionDirectory>())
                .onMultiSpans(spansSelector)
                .buildAndExecute(operationsDirectory)
        }
    }

    override fun cleanse(): Future<Boolean> =
        executor.submit(SRExecutionQueues.MonitoringStore) {
            "[Monitoring] Cleansing data store".logDWithThreadName(SR_LOG_TAG)
            OperationScopeBuilder(DeleteDirectory<SRMonitoringSessionDirectory>())
                .onMultiSpans(OldSpansSelector())
                .buildAndExecute(operationsDirectory); true
        }

    override fun onSpanEnded() {
        executor.execute(SRExecutionQueues.MonitoringStore) {
            "[Monitoring] Data store's running span is being ended.".logDWithThreadName(tag = SR_LOG_TAG)
            operationsDirectory = opsDirectoryFactory.apply { setCurrentSpanId(null) }.invoke()
        }
    }

    override fun shutdown(): Future<Boolean> =
        executor.submit(SRExecutionQueues.MonitoringStore) {
            "[Monitoring] Shutting down data store".logDWithThreadName(SR_LOG_TAG)
            OperationScopeBuilder(DeleteDirectory<SRMonitoringSessionDirectory>())
                .onMultiSpans(AllSpansSelector())
                .buildAndExecute(operationsDirectory)
            operationsDirectory =
                opsDirectoryFactory.apply { setCurrentSpanId(null) }.invoke(); true
        }
}