package com.instabug.library.sessionreplay

import androidx.annotation.VisibleForTesting
import com.instabug.library.OnSessionReplayLinkReady
import com.instabug.library.SessionSyncListener
import com.instabug.library.apichecker.APIChecker
import com.instabug.library.sessionreplay.di.SessionReplayServiceLocator
import com.instabug.library.sessionreplay.di.SessionReplayServiceLocator.sessionReplayDelegate
import com.instabug.library.util.extenstions.logError
import com.instabug.library.util.extenstions.logVerbose

private const val FEATURE_PREFIX = "%s for "

@VisibleForTesting
const val SR_ENABLING_ERROR =
    "Session Replay could not be enabled as it's currently disabled for your Instabug company account. Please contact customer support for further assistance."

@VisibleForTesting
const val SR_LOG_TYPE_ENABLING_SUCCESS = "%s enabled for Session Replay"
const val SR_ENABLING_SUCCESS = "Session Replay enabled"

@VisibleForTesting
const val SR_LOG_TYPE_DISABLING_SUCCESS = "%s disabled for Session Replay"
const val SR_DISABLING_SUCCESS = "Session Replay disabled"

object SessionReplay {
    /**
     * Enable or disable Session Replay.
     *
     * Use this method to enable or disable Session Replay.
     *
     * @param enabled Pass `true` to enable Session Replay, `false` to disable.
     * @see SessionReplay
     */
    @JvmStatic
    fun setEnabled(enabled: Boolean) {
        APIChecker.checkAndRunInExecutor("SessionReplay.setEnabled") {
            sessionReplayDelegate.handleRuntimeConfigurationsChange { config ->
                changeStateWithCheck(config.srAvailable, enabled) {
                    config.srEnabled = enabled
                }
            }
        }
    }

    /**
     * Enable or disable recording network logs for Session Replay.
     *
     * Use this method to control the recording of network logs during Session Replay.
     *
     * @param enable Pass `true` to enable recording of network logs, `false` to disable.
     * @see SessionReplay
     */
    @JvmStatic
    fun setNetworkLogsEnabled(enable: Boolean) {
        APIChecker.checkAndRunInExecutor("SessionReplay.setNetworkLogsEnabled") {
            sessionReplayDelegate.handleRuntimeConfigurationsChange { config ->
                changeStateWithCheck(config.networkLogsAvailable, enable, "Network logs") {
                    config.networkLogsRTEnabled = enable
                }
            }
        }
    }

    /**
     * Enable or disable recording Instabug logs for Session Replay.
     *
     * Use this method to control the recording of Instabug logs during Session Replay.
     *
     * @param enable Pass `true` to enable recording of Instabug logs, `false` to disable.
     * @see SessionReplay
     */
    @JvmStatic
    fun setIBGLogsEnabled(enable: Boolean) {
        APIChecker.checkAndRunInExecutor("SessionReplay.setIBGLogsEnabled") {
            sessionReplayDelegate.handleRuntimeConfigurationsChange { config ->
                changeStateWithCheck(config.ibgLogsAvailable, enable, "IBG logs") {
                    config.ibgLogsRTEnabled = enable
                }
            }
        }
    }

    /**
     * Enable or disable recording user steps for Session Replay.
     *
     * Use this method to control the recording of user steps during Session Replay.
     *
     * @param enable Pass `true` to enable recording of user steps, `false` to disable.
     * @see SessionReplay
     */
    @JvmStatic
    fun setUserStepsEnabled(enable: Boolean) {
        APIChecker.checkAndRunInExecutor("SessionReplay.setUserStepsEnabled") {
            sessionReplayDelegate.handleRuntimeConfigurationsChange { config ->
                changeStateWithCheck(config.userStepsAvailable, enable, "User steps") {
                    config.userStepsRTEnabled = enable
                }
            }
        }
    }

    /**
     * Retrieves current session's replay link.
     *
     * @param callback an [OnSessionReplayLinkReady] to be invoked when link is ready.
     * @see OnSessionReplayLinkReady
     */
    @JvmStatic
    fun getSessionReplayLink(callback: OnSessionReplayLinkReady) {
        runCatching {
            APIChecker.checkAndRunOrThrow("SessionReplay.getSessionReplayLink") {
                val replayLink = runCatching {
                    sessionReplayDelegate.getRunningSessionLink().get()
                }.onFailure { callback.onSessionReplayLinkReady(null) }.getOrThrow()
                callback.onSessionReplayLinkReady(replayLink)
            }
        }.onFailure { callback.onSessionReplayLinkReady(null) }
    }

    /**
     * Sets a callback to determine whether a Session Replay should be synchronized.
     *
     * This function allows you to register a [SessionSyncListener] to control the
     * synchronization of Session Replays. The provided listener will be invoked at the
     * beginning of the next session with metadata about the previous session.
     *
     * You can use this callback to implement custom logic to decide whether a Session
     * Replay should be synchronized based on specific criteria, such as session duration,
     * or session Attributes.
     *
     * @param sessionSyncListener The callback to be invoked before synchronizing a Session Replay.
     * If `null`, the default behavior is to synchronize all Session Replays.
     */
    @JvmStatic
    fun setSyncCallback(sessionSyncListener: SessionSyncListener?) {
        SessionReplayServiceLocator.userSessionSyncListener = sessionSyncListener
    }

    private fun changeStateWithCheck(
        isAvailable: Boolean,
        shouldEnabled: Boolean,
        logPrefix: String? = null,
        actionOnAvailable: () -> Unit
    ) {
        if (!isAvailable && shouldEnabled) {
            logPrefix
                ?.let { FEATURE_PREFIX.format(it) }
                ?.plus(SR_ENABLING_ERROR)
                ?.logError() ?: SR_ENABLING_ERROR.logError()
            return
        }
        actionOnAvailable()
        if (!shouldEnabled)
            logPrefix
                ?.let { SR_LOG_TYPE_DISABLING_SUCCESS.format(logPrefix) }
                ?.logVerbose()
                ?: SR_DISABLING_SUCCESS.logVerbose()
        else
            logPrefix
                ?.let { SR_LOG_TYPE_ENABLING_SUCCESS.format(logPrefix) }
                ?.logVerbose()
                ?: SR_ENABLING_SUCCESS.logVerbose()
    }
}