package com.instabug.library.model.session;

import android.annotation.SuppressLint;
import android.content.Context;

import androidx.annotation.Keep;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.BuildConfig;
import com.instabug.library.Constants;
import com.instabug.library.IBGFeature;
import com.instabug.library.Instabug;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.internal.device.InstabugDeviceProperties;
import com.instabug.library.internal.storage.cache.db.userAttribute.UserAttributesDbHelper;
import com.instabug.library.logging.InstabugUserEventLogger;
import com.instabug.library.model.UserAttributes;
import com.instabug.library.model.common.ICoreSession;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.tokenmapping.TokenMappingServiceLocator;
import com.instabug.library.user.UserEvent;
import com.instabug.library.user.UserManager;
import com.instabug.library.util.AppUtils;
import com.instabug.library.util.DeviceStateProvider;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.filters.AttributeFiltersFunctions;
import com.instabug.library.util.filters.Filters;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

@Keep
public class SessionLocalEntity {

    private final String id;
    private final String os;
    private final String device;
    private final long duration;
    private final long startedAt;
    private final String userName;
    private final String userEmail;
    private final String sdkVersion;
    private final String appVersion;
    private final String userEvents;
    private final String customAttributes;
    private final String userEventsKeys;
    private final String customAttributesKeys;
    private final boolean crashReportingEnabled;
    private final boolean isStitchedSessionLead;
    @SyncStatus
    private final int syncStatus;
    private final String uuid;
    private final String appToken;
    private final boolean usersPageEnabled;
    @Nullable
    private final String productionUsage;
    private final boolean isSessionV2Sent;


    public SessionLocalEntity(String id, String os, String device, long duration, long startedAt, String userName,
                              String userEmail, String sdkVersion, String appVersion, String userEvents,
                              String customAttributes, String userEventsKeys, String customAttributesKeys,
                              boolean crashReportingEnabled, @SyncStatus int syncStatus, String uuid,
                              String appToken, boolean usersPageEnabled, @Nullable String productionUsage,
                              boolean isStitchedSessionLead, boolean isSessionV2Sent) {
        this.id = id;
        this.os = os;
        this.device = device;
        this.duration = duration;
        this.startedAt = startedAt;
        this.userName = userName;
        this.userEmail = userEmail;
        this.sdkVersion = sdkVersion;
        this.appVersion = appVersion;
        this.userEvents = userEvents;
        this.customAttributes = customAttributes;
        this.userEventsKeys = userEventsKeys;
        this.customAttributesKeys = customAttributesKeys;
        this.crashReportingEnabled = crashReportingEnabled;
        this.syncStatus = syncStatus;
        this.uuid = uuid;
        this.appToken = appToken;
        this.usersPageEnabled = usersPageEnabled;
        this.productionUsage = productionUsage;
        this.isStitchedSessionLead = isStitchedSessionLead;
        this.isSessionV2Sent = isSessionV2Sent;
    }

    public String getUuid() {
        return uuid;
    }

    public String getAppToken() {
        return appToken;
    }

    public String getId() {
        return id;
    }

    public String getOs() {
        return os;
    }

    public String getDevice() {
        return device;
    }

    public long getDuration() {
        return duration;
    }

    public long getStartTimestampMicros() {
        return startedAt;
    }

    public String getUserName() {
        return userName;
    }

    public String getUserEmail() {
        return userEmail;
    }

    public String getSdkVersion() {
        return sdkVersion;
    }

    public String getAppVersion() {
        return appVersion;
    }

    public String getUserEvents() {
        return userEvents;
    }

    public String getCustomAttributes() {
        return customAttributes;
    }

    public String getUserEventsKeys() {
        return userEventsKeys;
    }

    public String getCustomAttributesKeys() {
        return customAttributesKeys;
    }

    public boolean isCrashReportingEnabled() {
        return crashReportingEnabled;
    }

    public boolean isStitchedSessionLead() {
        return isStitchedSessionLead;
    }

    @SyncStatus
    public int getSyncStatus() {
        return syncStatus;
    }

    public boolean isUsersPageEnabled() {
        return usersPageEnabled;
    }

    public boolean isSessionV2Sent() {
        return isSessionV2Sent;
    }

    public static class Factory {

        @NonNull
        @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
        public SessionLocalEntity create(Context context, @NonNull ICoreSession session, boolean usersPageEnabled, boolean isSessionV2Sent) {
            return new SessionLocalEntity(
                    session.getId(),
                    getOs(usersPageEnabled),
                    getDevice(usersPageEnabled),
                    getCurrentSessionTimeUntilNow(),
                    getSessionStartedAt(),
                    getUserName(usersPageEnabled),
                    getUserEmail(usersPageEnabled),
                    getSdkVersion(),
                    getAppVersion(context),
                    getUserEventsAsStringJsonArray(),
                    getUserAttributesAsStringJsonObject(),
                    getUserEventsKeysAsStringJsonArray(),
                    getUserAttributesKeysAsStringJsonArray(),
                    isCrashReportingEnabled(),
                    getSyncStatus(),
                    getUuid(),
                    getAppToken(),
                    usersPageEnabled,
                    getProductionUsage(),
                    session.isStitchedSessionLead(),
                    isSessionV2Sent);
        }

        @Nullable
        private String getProductionUsage() {
            if (InstabugCore.isFeatureAvailable(IBGFeature.PRODUCTION_USAGE_DETECTION)) {
                try {
                    JSONObject productionUsage = new JSONObject();
                    //App installer package
                    String installerPackageName = AppUtils.getInstallerPackageName(Instabug.getApplicationContext());
                    if (installerPackageName != null) {
                        productionUsage.put("store_url", installerPackageName);
                    } else {
                        productionUsage.put("store_url", "other");
                    }
                    // Features wrapper
                    JSONObject featuresJsonObject = new JSONObject();
                    // Feature bug reporting
                    JSONObject bugReportingJsonObject = new JSONObject();
                    bugReportingJsonObject.put("enabled",
                            InstabugCore.isFeatureEnabled(IBGFeature.BUG_REPORTING));
                    featuresJsonObject.put("bugs", bugReportingJsonObject);
                    // Feature surveys
                    JSONObject surveysJsonObject = new JSONObject();
                    surveysJsonObject.put("enabled",
                            InstabugCore.isFeatureEnabled(IBGFeature.SURVEYS));
                    featuresJsonObject.put("surveys", surveysJsonObject);
                    // Feature requests
                    JSONObject featureRequestsJsonObject = new JSONObject();
                    featureRequestsJsonObject.put("enabled",
                            InstabugCore.isFeatureEnabled(IBGFeature.FEATURE_REQUESTS));
                    featuresJsonObject.put("feature_requests", featureRequestsJsonObject);
                    // Feature APM
                    JSONObject apmJsonObject = new JSONObject();
                    apmJsonObject.put("enabled", InstabugCore.isAPMEnabled());
                    featuresJsonObject.put("apm", apmJsonObject);
                    productionUsage.put("features", featuresJsonObject);
                    return productionUsage.toString();
                } catch (JSONException e) {
                    return null;
                }
            }
            return null;
        }

        private boolean isCrashReportingEnabled() {
            return InstabugCore.isCrashReportingEnabled();
        }

        @Nullable
        private String getOs(boolean usersPageEnabled) {
            return usersPageEnabled ? "SDK Level " + InstabugDeviceProperties.getCurrentOSLevel() : null;
        }

        @Nullable
        private String getDevice(boolean usersPageEnabled) {
            return usersPageEnabled ? InstabugDeviceProperties.getDeviceType() : null;
        }

        private long getCurrentSessionTimeUntilNow() {
            long sessionStartedAt = SettingsManager.getInstance().getSessionStartedAt();
            if (sessionStartedAt == 0L) return 0L;
            else return (System.currentTimeMillis() / 1000L) - sessionStartedAt;
        }

        private long getSessionStartedAt() {
            return SettingsManager.getInstance().getSessionStartedAt();
        }

        @Nullable
        private String getUserName(boolean usersPageEnabled) {
            return usersPageEnabled ? UserManager.getIdentifiedUsername() : null;
        }

        @Nullable
        private String getUserEmail(boolean usersPageEnabled) {
            return usersPageEnabled ? UserManager.getIdentifiedUserEmail() : null;
        }

        @NonNull
        private String getSdkVersion() {
            return BuildConfig.SDK_VERSION;
        }

        @Nullable
        private String getAppVersion(Context context) {
            return context == null ? null : DeviceStateProvider.getAppVersion(context);
        }

        @NonNull
        private String getUserAttributesAsStringJsonObject() {
            String userAttributesString = "{}";
            HashMap<String, String> attributesMap = Filters.applyOn(UserAttributesDbHelper.getAll())
                    .apply(AttributeFiltersFunctions.nonBackEndMapFilter())
                    .thenGet();
            if (attributesMap != null && attributesMap.size() != 0) {
                UserAttributes userAttributes = new UserAttributes();
                userAttributes.putMap(attributesMap);
                userAttributesString = userAttributes.toString();
            }
            return userAttributesString;
        }

        @NonNull
        private String getUserAttributesKeysAsStringJsonArray() {
            String userAttributesString = "[]";
            HashMap<String, String> attributesMap = Filters.applyOn(UserAttributesDbHelper.getAll())
                    .apply(AttributeFiltersFunctions.nonBackEndMapFilter())
                    .thenGet();
            if (attributesMap != null && attributesMap.size() != 0) {
                UserAttributes userAttributes = new UserAttributes();
                userAttributes.putMap(attributesMap);
                userAttributesString = userAttributes.keysAsStringJsonArray();
            }
            return userAttributesString;
        }

        @NonNull
        private String getUserEventsAsStringJsonArray() {
            String userEventsAsString = "[]";
            try {
                List<UserEvent> userEvents = InstabugUserEventLogger.getInstance().getUserEvents();
                //copying user events into new list to avoid concurrency
                List<UserEvent> userEventList = new ArrayList<>(userEvents);
                userEventsAsString = UserEvent.toJson(userEventList).toString();
            } catch (JSONException e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "parsing user events got error: " + e.getMessage(), e);
            }
            return userEventsAsString;
        }

        @NonNull
        private String getUserEventsKeysAsStringJsonArray() {
            List<UserEvent> userEvents = InstabugUserEventLogger.getInstance().getUserEvents();
            //copying user events into new list to avoid concurrency
            List<UserEvent> userEventList = new ArrayList<>(userEvents);
            return UserEvent.keysAsJsonArray(userEventList).toString();
        }

        private boolean isFirstSession() {
            return SettingsManager.getInstance().isFirstSession();
        }

        private int getSyncStatus() {
            return isFirstSession() ? SyncStatus.READY_FOR_SYNC : SyncStatus.OFFLINE;
        }

        @NonNull
        private String getUuid() {
            return UserManager.getUUID();
        }

        @Nullable
        private String getAppToken() {
            return TokenMappingServiceLocator.getTokenMappingConfigs().getAvailableAppToken();
        }
    }

    @Nullable
    public String getProductionUsage() {
        return productionUsage;
    }
}
