package com.instabug.library.internal.storage;

import android.annotation.SuppressLint;
import android.content.Context;
import android.graphics.Bitmap;
import android.net.Uri;
import android.os.Environment;

import androidx.annotation.NonNull;

import com.instabug.library.Constants;
import com.instabug.library.Instabug;
import com.instabug.library.internal.servicelocator.CoreServiceLocator;
import com.instabug.library.internal.storage.executor.DeleteOperationExecutor;
import com.instabug.library.internal.storage.executor.ReadOperationExecutor;
import com.instabug.library.internal.storage.executor.WriteOperationExecutor;
import com.instabug.library.internal.storage.operation.DiskOperation;
import com.instabug.library.sessionreplay.di.SessionReplayServiceLocator;
import com.instabug.library.util.FileUtils;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.extenstions.CtxKt;
import com.instabug.library.util.threading.PoolProvider;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.concurrent.ExecutionException;

import kotlin.io.FilesKt;


public final class DiskUtils {
    public static final String TAG = "DiskUtils";
    private final Context context;

    private DiskUtils(Context context) {
        this.context = context;
    }

    public static File getInstabugInternalDirectory(Context context) {
        String directoryPath = getInternalStoragePath(context);
        File directory = new File(directoryPath + "/instabug/");
        if (!directory.exists()) {
            if (directory.mkdirs()) {
                File noMediaFile = new File(directory, ".nomedia");
                try {
                    noMediaFile.createNewFile();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        return directory;
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public static File getInstabugDirectory(Context context) {
        String directoryPath;
        if (context.getExternalFilesDir(null) != null
                && Environment.getExternalStorageState().equals(Environment.MEDIA_MOUNTED)) {
            try {
                directoryPath = context.getExternalFilesDir(null).getAbsolutePath();
            } catch (NullPointerException npe) {
                directoryPath = getInternalStoragePath(context);
                InstabugSDKLogger.w(TAG, "External storage not available, " +
                        "saving file to internal storage.");
            }
        } else {
            directoryPath = getInternalStoragePath(context);
            InstabugSDKLogger.i(TAG, "External storage not available, " +
                    "saving file to internal storage.");
        }
        File directory = new File(directoryPath + "/instabug/");
        if (!directory.exists()) {
            if (directory.mkdirs()) {
                File noMediaFile = new File(directory, ".nomedia");
                try {
                    noMediaFile.createNewFile();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        return directory;
    }

    /**
     * A method to get the log directory in the external storage
     *
     * @param directoryLogName the directory name
     * @return
     */
    public static synchronized File getInsatbugLogDirectory(String directoryLogName, Context context) {
        File externalStorageDirectory = getInstabugDirectory(context);
        String externalStorageDirectoryPath = externalStorageDirectory.getAbsolutePath().concat("/");
        File logsDirectory = new File(externalStorageDirectoryPath.concat(directoryLogName));
        if (!logsDirectory.exists()) {
            boolean newFile = logsDirectory.mkdir();
            if (!newFile)
                InstabugSDKLogger.w(TAG,
                        "failed logs directory to create");
        }
        return logsDirectory;
    }

    public static String getInternalStoragePath(Context context) {
        return CtxKt.getInternalFilesDir(context).getAbsolutePath();
    }

    public static void saveBitmapOnDisk(Bitmap bitmap, File file) throws IOException {
        if (bitmap != null && file != null) {
            InstabugSDKLogger.v(Constants.LOG_TAG, "starting save viewHierarchy image, path: " +
                    file.getAbsolutePath());
            FileOutputStream outputStream = new FileOutputStream(file);
            bitmap.compress(Bitmap.CompressFormat.PNG, 100, outputStream);
            outputStream.close();
            InstabugSDKLogger.v(Constants.LOG_TAG, "viewHierarchy image saved, path: " + file
                    .getAbsolutePath() + ", time in MS: " + System.currentTimeMillis());
        }
    }

    public static void cleanDirectory(File directory) {
        if (directory.exists() && directory.isDirectory()) {
            File[] files = directory.listFiles();
            if (files != null) {
                for (File child : files) {
                    if (child.delete()) {
                        InstabugSDKLogger.v(Constants.LOG_TAG, "file deleted successfully, path: "
                                + child.getPath());
                    }
                }
            }
        }
    }

    @SuppressLint("RESOURCE_LEAK")
    public static void copyFromUriIntoFile(@NonNull Context context, Uri source, File target) throws
            IOException {
        BufferedInputStream bufferedInputStream = null;
        BufferedOutputStream bufferedOutputStream = null;
        if (source != null) {
            InputStream in = context.getContentResolver().openInputStream(source);
            if (in != null) {
                bufferedInputStream = new BufferedInputStream(in);
            }
        }
        if (bufferedInputStream != null) {
            InstabugSDKLogger.d(TAG, "Target file path: " + target.getPath());
            Uri targetUri = Uri.fromFile(target);
            if (targetUri != null) {
                OutputStream out = context.getContentResolver().openOutputStream(targetUri);
                if (out != null) {
                    bufferedOutputStream = new BufferedOutputStream(out);
                }
            }
            if (bufferedOutputStream != null) {
                byte[] buff = new byte[32 * 1024];
                int len;
                while ((len = bufferedInputStream.read(buff)) > 0) {
                    bufferedOutputStream.write(buff, 0, len);
                }
            }
        }
        try {
            if (bufferedOutputStream != null) {
                bufferedOutputStream.close();
            }
            if (bufferedInputStream != null) {
                bufferedInputStream.close();
            }
        } catch (IOException e) {
            if (e.getMessage() != null) {
                InstabugSDKLogger.e(TAG, e.getMessage(), e);
            }
        }
    }

    public static void deleteFile(final String fileLocalPath) {
        PoolProvider.postIOTask(() -> {
            File file = new File(fileLocalPath);
            file.delete();
        });
    }

    public static DiskUtils with(Context context) {
        return new DiskUtils(context);
    }

    public static boolean isFileExist(String path) {
        return new File(path).exists();
    }

    public static void deleteSDKDirectory() {
        deleteSDKDataHub();
        Context context = Instabug.getApplicationContext();
        if (context != null) {
            File sdkDirectory = getInstabugInternalDirectory(context);
            if (sdkDirectory.exists()) {
                FileUtils.deleteDirectory(sdkDirectory);
                sdkDirectory.delete();
            }
        }
    }

    public static void deleteSDKDataHub() {
        try {
            CoreServiceLocator.getDataHubController().shutdown().get();
        } catch (InterruptedException e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error deleting SDK data hub directory [Thread interrupted]", e);
            Thread.currentThread().interrupt();
        } catch (ExecutionException e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error deleting SDK data hub directory", e);
        }
    }

    public static void deleteSdkExternalDirectory() {
        Context context = Instabug.getApplicationContext();
        if (context != null) {
            File externalStorageDir = getInstabugDirectory(context);
            if (externalStorageDir.exists()) {
                FileUtils.deleteDirectory(externalStorageDir);
                externalStorageDir.delete();
            }
        }
    }

    public ReadOperationExecutor readOperation
            (@NonNull DiskOperation<String, Void> operation) {
        return new ReadOperationExecutor(operation);
    }

    public WriteOperationExecutor writeOperation
            (@NonNull DiskOperation<Uri, Context> operation) {
        return new WriteOperationExecutor(context, operation);
    }

    public DeleteOperationExecutor deleteOperation
            (@NonNull DiskOperation<Boolean, Void> operation) {
        return new DeleteOperationExecutor(operation);
    }

    public static File createStateTextFile(Context context, String fileNamePrefix) {
        return new File(context.getFilesDir()
                .getAbsolutePath()
                + File.pathSeparator
                + fileNamePrefix
                + File.pathSeparator
                + System.currentTimeMillis() + ".txt");
    }

    public static void deleteCrashStateFiles() {
        deleteStateFiles("crash_state");
    }

    public static void deleteNonfatalStateFiles() {
        deleteStateFiles("non_fatal_state");
    }

    public static void deleteBugsStateFiles() {
        deleteStateFiles("bug_state");
    }

    public static void deleteAppTerminationsStateFiles() {
        deleteStateFiles("app_termination_state");
    }

    public static void deleteAllStateFiles() {
        Context context = Instabug.getApplicationContext();
        if (context != null) {
            File rootDirectory = context.getFilesDir();
            File parent = rootDirectory.getParentFile();
            FilenameFilter filter = (dir, name) ->
                    name.startsWith("files:") && name.endsWith(".txt") && !name.contains("non_fatal_state");
            if (parent != null) {
                File[] stateFiles = parent.listFiles(filter);
                if (stateFiles != null) {
                    for (File stateFile : stateFiles) {
                        stateFile.delete();
                    }
                }
            }
        }
    }

    public static void deleteSessionReplayDirectory() {
        Context context = Instabug.getApplicationContext();
        if (context == null) return;
        File instabugDirectory = new File(getInternalStoragePath(context) + "/instabug/");
        if (!instabugDirectory.exists()) return;
        FilenameFilter filter = (dir, name) -> name.equals("session-replay");
        File[] filterResults = instabugDirectory.listFiles(filter);
        if (filterResults != null && filterResults.length > 0) {
            FilesKt.deleteRecursively(filterResults[0]);
        }
    }

    public static void cleanseSRMonitoringDirectory() {
        SessionReplayServiceLocator.getMonitorDelegate().onBeingCleansed();
    }

    public static void deleteStateFiles(String filePrefix) {
        Context context = Instabug.getApplicationContext();
        if (context != null) {
            File rootDirectory = context.getFilesDir();
            File parent = rootDirectory.getParentFile();
            FilenameFilter filter = (dir, name) -> name.startsWith("files:" + filePrefix) && name.endsWith(".txt");
            if (parent != null) {
                File[] stateFiles = parent.listFiles(filter);
                if (stateFiles != null) {
                    for (File stateFile : stateFiles) {
                        stateFile.delete();
                    }
                }
            }
        }
    }
}
