package com.instabug.library.internal.servicelocator

import android.app.Application
import android.content.Context
import android.content.SharedPreferences
import android.view.View
import androidx.annotation.VisibleForTesting
import com.instabug.library.AppLaunchIDProvider
import com.instabug.library.Constants
import com.instabug.library.Instabug
import com.instabug.library.InstabugFeaturesManager
import com.instabug.library.PlatformMapper
import com.instabug.library.R
import com.instabug.library.WatchableSpansCacheDirectory
import com.instabug.library.core.eventbus.ActivityLifecycleSubscriber
import com.instabug.library.core.eventbus.ActivityLifecycleSubscriberImpl
import com.instabug.library.core.eventbus.DefaultActivityLifeCycleEventHandler
import com.instabug.library.core.eventbus.eventpublisher.IBGEventBusExceptionHandler
import com.instabug.library.core.eventbus.eventpublisher.IBGEventBusExceptionHandlerImpl
import com.instabug.library.datahub.DefaultHubController
import com.instabug.library.datahub.HubControlledDataStore
import com.instabug.library.datahub.HubController
import com.instabug.library.datahub.HubDirectory
import com.instabug.library.datahub.HubExecutionQueues
import com.instabug.library.datahub.HubGarbageCollector
import com.instabug.library.datahub.IBGLogsHubDataStore
import com.instabug.library.datahub.NetworkLogsHubDataStore
import com.instabug.library.datahub.SimpleHubDataReceiver
import com.instabug.library.diagnostics.IBGDiagnostics
import com.instabug.library.featuresflags.configs.CompositeFeatureFlagsConfigurationHandler
import com.instabug.library.featuresflags.configs.FeatureFlagsConfigsHandler
import com.instabug.library.interactionstracking.AndroidViewsTargetUILocator
import com.instabug.library.interactionstracking.IBGUINode
import com.instabug.library.interactionstracking.LegacyViewUINodeTransformer
import com.instabug.library.interactionstracking.TargetUILocator
import com.instabug.library.interactionstracking.UINodeTransformer
import com.instabug.library.internal.sdkexperiments.SDKExperiment
import com.instabug.library.internal.sdkexperiments.SDKExperimentConfigurator
import com.instabug.library.internal.sdkexperiments.SDKExperimentsManager
import com.instabug.library.internal.sharedpreferences.InstabugSharedPreferences
import com.instabug.library.internal.storage.AttachmentManager
import com.instabug.library.internal.storage.cache.db.DatabaseManager
import com.instabug.library.internal.storage.cache.db.SQLiteDatabaseWrapper
import com.instabug.library.invocation.InvocationManagerContract
import com.instabug.library.logging.InstabugLog.LogMessage
import com.instabug.library.logscollection.DataWatcher
import com.instabug.library.logscollection.LogDistributor
import com.instabug.library.logscollection.SimpleDataWatcher
import com.instabug.library.map.Mapper
import com.instabug.library.model.NetworkLog
import com.instabug.library.networkDiagnostics.caching.NetworkDiagnosticsCachingManager
import com.instabug.library.networkDiagnostics.caching.NetworkDiagnosticsCachingManagerImpl
import com.instabug.library.networkDiagnostics.configuration.NetworkDiagnosticsConfigurationHandler
import com.instabug.library.networkDiagnostics.configuration.NetworkDiagnosticsConfigurationHandlerImpl
import com.instabug.library.networkDiagnostics.configuration.NetworkDiagnosticsConfigurationProvider
import com.instabug.library.networkDiagnostics.configuration.NetworkDiagnosticsConfigurationProviderImpl
import com.instabug.library.networkDiagnostics.manager.NetworkDiagnosticsManager
import com.instabug.library.networkDiagnostics.manager.NetworkDiagnosticsManagerImpl
import com.instabug.library.networkv2.authorization.ispx.ISP
import com.instabug.library.networkv2.authorization.ispx.ISPSDKExperiment
import com.instabug.library.performanceclassification.DevicePerformanceClassConfig
import com.instabug.library.performanceclassification.DevicePerformanceClassConfigImpl
import com.instabug.library.performanceclassification.DevicePerformanceClassHelper
import com.instabug.library.screenshot.ScreenshotProvider
import com.instabug.library.screenshot.analytics.CommonAnalyticsCollector
import com.instabug.library.screenshot.analytics.ScreenShotAnalyticsMapper
import com.instabug.library.screenshot.analytics.ScreenshotAnalytics
import com.instabug.library.screenshot.instacapture.ComposedViewsFilter
import com.instabug.library.screenshot.instacapture.IBGScreenshotCapture
import com.instabug.library.screenshot.instacapture.IBGScreenshotCaptureImpl
import com.instabug.library.screenshot.instacapture.PrivateViewsFilter
import com.instabug.library.screenshot.instacapture.UserMaskingFilterProvider
import com.instabug.library.screenshot.instacapture.UserMaskingFilterProviderImpl
import com.instabug.library.screenshot.instacapture.ViewsFilter
import com.instabug.library.screenshot.subscribers.ScreenshotsAnalyticsEventBus
import com.instabug.library.settings.PersistableSettings
import com.instabug.library.tracking.ActiveScreenEvaluator
import com.instabug.library.tracking.ComposeLifeCycleMonitor
import com.instabug.library.tracking.IBGActivityLifecycleMonitor
import com.instabug.library.tracking.IBGComposeLifeCycleMonitor
import com.instabug.library.tracking.InstabugScreenOffEventMonitor
import com.instabug.library.tracking.LatestVisibleEvaluator
import com.instabug.library.tracking.NavigableViewsTrackingDelegate
import com.instabug.library.tracking.ScreenOffHandler
import com.instabug.library.tracking.ScreensTreeHandler
import com.instabug.library.tracking.StartedActivitiesCounter
import com.instabug.library.tracking.StartedActivitiesCounterSingleton
import com.instabug.library.user.EmailValidator
import com.instabug.library.user.UserIdValidator
import com.instabug.library.user.Validator
import com.instabug.library.util.CoolDownThrottle
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.LimitConstraintApplier
import com.instabug.library.util.LimitConstraintsApplierImpl
import com.instabug.library.util.SdkDataCleaningUtil
import com.instabug.library.util.threading.IBGQueueMonitoringHelper
import com.instabug.library.util.threading.OrderedExecutorService
import com.instabug.library.util.threading.PoolProvider
import com.instabug.library.util.threading.ReturnableSingleThreadExecutor
import com.instabug.library.visualusersteps.BasicReproScreenshotsCapturingProxy
import com.instabug.library.visualusersteps.BasicReproStepsCapturingProxy
import com.instabug.library.visualusersteps.CompositeReproCapturingProxy
import com.instabug.library.visualusersteps.ReproCapturingProxy
import com.instabug.library.visualusersteps.ReproScreenshotsCacheDirectory
import com.instabug.library.visualusersteps.ReproScreenshotsCapturingProxy
import com.instabug.library.visualusersteps.ReproStepsCaptor
import com.instabug.library.visualusersteps.ReproStepsCapturingProxy
import com.instabug.library.visualusersteps.ReproStepsManualCaptor
import com.instabug.library.visualusersteps.TouchedViewExtractor
import com.instabug.library.visualusersteps.VisualUserStepsProvider
import com.instabug.library.visualusersteps.manual.ReproStepsManualCaptorProxy
import com.instabug.library.visualusersteps.manual.configuration.ManualUserStepsConfigurationHandler
import com.instabug.library.visualusersteps.manual.configuration.ManualUserStepsConfigurationProvider
import com.instabug.library.visualusersteps.manual.configuration.ManualUserStepsConfigurationProviderImpl
import org.json.JSONObject

object CoreServiceLocator {
    private const val ATTRIBUTES_CHARACTERS_LIMIT = 90
    private const val ATTRIBUTES_STORE_LIMIT = 100
    private var screenOffHandler: ScreenOffHandler? = null

    private var invocationManagerContract: InvocationManagerContract? = null


    @JvmStatic
    fun getInstabugSharedPreferences(context: Context, name: String): SharedPreferences? {
        val preferences = getSharedPreferencesExecutor()
            .executeAndGet { InstabugSharedPreferences.create(context, name) }
        if (preferences == null) {
            IBGDiagnostics.reportNonFatal(
                NullSharedPreferencesException(),
                "Trying to access sharedPref while being NULL"
            )
        }
        return preferences
    }

    @JvmStatic
    fun getSharedPreferencesExecutor(): ReturnableSingleThreadExecutor =
        PoolProvider.getReturnableSingleThreadExecutor("SharedPrefs")

    @JvmStatic
    fun getScreenOffEventMonitor(application: Application): InstabugScreenOffEventMonitor {
        return InstabugScreenOffEventMonitor(application)
    }

    fun getQueueMonitoringHelper(): IBGQueueMonitoringHelper =
        IBGQueueMonitoringHelper(
            PersistableSettings.getInstance()?.dequeueThreshold ?: 0L,
            PersistableSettings.getInstance()?.completionThreshold ?: 0L
        )

    @JvmStatic
    @Synchronized
    fun getDatabaseManager(): DatabaseManager? = kotlin.runCatching {
        DatabaseManager.getInstance()
    }.getOrElse { throwable ->
        InstabugSDKLogger.e(Constants.LOG_TAG, "Couldn't open database.")
        IBGDiagnostics.reportNonFatal(throwable, "Couldn't open database.")
        return@getOrElse null
    }

    @JvmStatic
    @Synchronized
    fun getDatabaseWrapper(): SQLiteDatabaseWrapper? = getDatabaseManager()?.openDatabase()

    @JvmStatic
    @Synchronized
    fun getScreenOffHandler(): ScreenOffHandler {
        if (screenOffHandler == null) {
            screenOffHandler = ScreenOffHandler()
        }
        return screenOffHandler!!
    }

    @JvmStatic
    fun getSdkCleaningUtil(): SdkDataCleaningUtil {
        return SdkDataCleaningUtil()
    }

    @JvmStatic
    fun createActivityLifecycleSubscriber(eventHandler: DefaultActivityLifeCycleEventHandler): ActivityLifecycleSubscriber =
        ActivityLifecycleSubscriberImpl(eventHandler)

    @JvmStatic
    val eventBusExceptionHandler: IBGEventBusExceptionHandler
        get() = IBGEventBusExceptionHandlerImpl

    @JvmStatic
    val screensRoot: ScreensTreeHandler by lazy { ScreensTreeHandler() }

    @JvmStatic
    val screenActivityComponentsMonitor: IBGActivityLifecycleMonitor
            by lazy { IBGActivityLifecycleMonitor(screensRoot) }

    @JvmStatic
    val composeLifeCycleMonitor: ComposeLifeCycleMonitor by lazy {
        IBGComposeLifeCycleMonitor(screensRoot, reproStepsProxy)
    }

    @JvmStatic
    val navigableViewsTracker: NavigableViewsTrackingDelegate by lazy { NavigableViewsTrackingDelegate() }

    @JvmStatic
    val activeScreenEvaluator: ActiveScreenEvaluator
        get() = LatestVisibleEvaluator

    @JvmStatic
    val userMaskingFilterProvider: UserMaskingFilterProvider
        get() = UserMaskingFilterProviderImpl

    @JvmStatic
    val composedViewsFilter: ViewsFilter
        get() = mutableSetOf<ViewsFilter>(PrivateViewsFilter)
            .apply { addAll(userMaskingFilterProvider.userMaskingFilters) }
            .let(::ComposedViewsFilter)

    @JvmStatic
    val ignoredViewsIds: IntArray = intArrayOf(
        R.id.instabug_decor_view,
        R.id.instabug_extra_screenshot_button,
        R.id.instabug_floating_button,
        R.id.instabug_in_app_notification,
        R.id.instabug_intro_dialog
    )

    @JvmStatic
    val screenshotCapture: IBGScreenshotCapture
        get() = IBGScreenshotCaptureImpl

    @JvmStatic
    val commonAnalyticsCollector: CommonAnalyticsCollector by lazy {
        CommonAnalyticsCollector()
    }

    @JvmStatic
    val screenshotsAnalyticsEventBus: ScreenshotsAnalyticsEventBus
            by lazy {
                ScreenshotsAnalyticsEventBus
            }
    val orderedExecutor: OrderedExecutorService
        get() = PoolProvider.getInstance().orderedExecutor

    @JvmStatic
    val instabugFeaturesManager: InstabugFeaturesManager
        get() = InstabugFeaturesManager.getInstance()

    @JvmStatic
    val reproScreenshotsProxy: ReproScreenshotsCapturingProxy
            by lazy {
                BasicReproScreenshotsCapturingProxy(
                    originalCaptor = ScreenshotProvider,
                    savingDirectory = reproScreenshotsCacheDir,
                    executor = orderedExecutor
                )
            }

    @VisibleForTesting
    @JvmStatic
    val visualUserStepsProvider: ReproStepsCaptor
        get() = VisualUserStepsProvider.getInstance(
            PoolProvider.getSingleThreadExecutor(VISUAL_USER_STEPS_PROVIDER_EXECUTOR)
        )

    @JvmStatic
    val reproStepsProxy: ReproStepsCapturingProxy
            by lazy { BasicReproStepsCapturingProxy(visualUserStepsProvider, orderedExecutor) }

    @JvmStatic
    val reproCompositeProxy: ReproCapturingProxy by lazy {
        CompositeReproCapturingProxy(listOf(reproStepsProxy, reproScreenshotsProxy))
    }

    @JvmStatic
    val reproScreenshotsCacheDir: WatchableSpansCacheDirectory by lazy {
        ReproScreenshotsCacheDirectory(
            executor = orderedExecutor,
            spanIDProvider = AppLaunchIDProvider,
            ctxGetter = Instabug::getApplicationContext,
            baseDirectoryGetter = AttachmentManager::getAttachmentInternalDirectory
        )
    }

    @JvmStatic
    fun getUserAttributesStoreLimit() = ATTRIBUTES_STORE_LIMIT

    @JvmStatic
    fun getUserAttributesCharactersLimit() = ATTRIBUTES_CHARACTERS_LIMIT

    @JvmStatic
    fun getDevicePerformanceClassConfig(): DevicePerformanceClassConfig =
        DevicePerformanceClassConfigImpl

    fun getDevicePerformanceClassHelper() = DevicePerformanceClassHelper()

    @JvmStatic
    fun getLimitConstraintApplier(): LimitConstraintApplier = LimitConstraintsApplierImpl(
        getDevicePerformanceClassHelper()
    )

    val userIdValidator: Validator by lazy {
        UserIdValidator()
    }
    val emailValidator: Validator by lazy {
        EmailValidator()
    }

    @JvmStatic
    fun getInvocationManagerContract(): InvocationManagerContract? = invocationManagerContract

    @JvmStatic
    fun setInvocationManagerContract(contract: InvocationManagerContract) {
        this.invocationManagerContract = contract
    }

    @JvmStatic
    val networkLogsDistributor: LogDistributor<NetworkLog> by lazy {
        LogDistributor.createBasicLogDistributorFor(SimpleHubDataReceiver(networkLogStore))
    }

    @JvmStatic
    val ibgLogsDistributor: LogDistributor<LogMessage> by lazy {
        LogDistributor.createBasicLogDistributorFor(SimpleHubDataReceiver(ibgLogStore))
    }

// region Data Hub

    @JvmStatic
    val dataHubController: HubController by lazy {
        DefaultHubController(
            orderedExecutor,
            HubDirectory.Factory(
                Instabug::getApplicationContext,
                AttachmentManager::getAttachmentInternalDirectory
            ),
            dataStores
        )
    }

    val dataStores: List<HubControlledDataStore<*>>
        get() = listOf(networkLogStore, ibgLogStore)

    val networkLogStore: HubControlledDataStore<NetworkLog>
            by lazy {
                NetworkLogsHubDataStore(
                    executor = orderedExecutor,
                    batcher = NetworkLogsHubDataStore.createBatcher(getLimitConstraintApplier())
                )
            }

    @JvmStatic
    val ibgLogStore: HubControlledDataStore<LogMessage>
            by lazy {
                IBGLogsHubDataStore(
                    executor = orderedExecutor,
                    batcher = IBGLogsHubDataStore.createBatcher(getLimitConstraintApplier())
                )
            }

    @JvmStatic
    val hubDataWatcher: DataWatcher by lazy {
        SimpleDataWatcher(
            executor = orderedExecutor,
            collector = HubGarbageCollector(dataHubController),
            executionQueue = HubExecutionQueues.Controller
        )
    }

    //region Views targeting and labeling
    @JvmStatic
    var touchedViewExtractorExtension: TouchedViewExtractor? = null

    @JvmStatic
    val targetUILocator: TargetUILocator<Pair<IBGUINode, String>>
            by lazy { AndroidViewsTargetUILocator { uiNodesTransformer } }

    @get:JvmStatic
    var uiNodesTransformer: UINodeTransformer<View> = LegacyViewUINodeTransformer

// endregion

    //region Network Diagnostics
    @JvmStatic
    val networkDiagnosticsConfigurationHandler: NetworkDiagnosticsConfigurationHandler
        get() = NetworkDiagnosticsConfigurationHandlerImpl()


    @JvmStatic
    val networkDiagnosticsConfigurationProvider: NetworkDiagnosticsConfigurationProvider
        get() = NetworkDiagnosticsConfigurationProviderImpl()


    val networkDiagnosticsCachingManager: NetworkDiagnosticsCachingManager
        get() = NetworkDiagnosticsCachingManagerImpl(Instabug::getApplicationContext)


    @JvmStatic
    val networkDiagnosticsManager: NetworkDiagnosticsManager by lazy {
        NetworkDiagnosticsManagerImpl(
            orderedExecutor,
            networkDiagnosticsConfigurationProvider,
            networkDiagnosticsCachingManager,
            PoolProvider.getInstance().scheduledExecutor,
            PoolProvider.getInstance().mainExecutor
        )
    }

    @JvmStatic
    val screenShotAnalyticsMapper: Mapper<ScreenshotAnalytics, JSONObject> by lazy { ScreenShotAnalyticsMapper() }
//endregion

    // region platform mapper
    val platformMapper: Mapper<Pair<String, Int>, Int>
        get() = PlatformMapper()
//endregion

    @JvmStatic
    val startedActivitiesCounter: StartedActivitiesCounter
        get() = StartedActivitiesCounterSingleton

    @JvmStatic
    val compositeFeatureFlagHandler: FeatureFlagsConfigsHandler
        get() = CompositeFeatureFlagsConfigurationHandler(
            listOf(
                Provider { manualUserStepsConfigurationHandler }
            )
        )

    // region ManualUserSteps

    private const val MANUAL_SCREENSHOT_COOLDOWN_TIME = 1000L

    private val manualUserStepsConfigurationProvider: ManualUserStepsConfigurationProvider by lazy {
        ManualUserStepsConfigurationProviderImpl()
    }

    private val manualUserStepsConfigurationHandler: FeatureFlagsConfigsHandler
        get() = ManualUserStepsConfigurationHandler(manualUserStepsConfigurationProvider)

    @JvmStatic
    val reproStepsManualCaptor: ReproStepsManualCaptor by lazy {
        ReproStepsManualCaptorProxy(
            reproStepsProxy,
            manualUserStepsConfigurationProvider,
            CoolDownThrottle(MANUAL_SCREENSHOT_COOLDOWN_TIME)
        )
    }

    // endregion

    // region Signing Experiment
    private val signingSDKExperiment: SDKExperiment<ISP> by lazy { ISPSDKExperiment() }

    @get:JvmStatic
    val signingImplementation: ISP by lazy { signingSDKExperiment.getImplementation() }
    // endregion

    // region SDK Experiments
    @get:JvmStatic
    val sdkExperimentsManager: SDKExperimentConfigurator by lazy {
        listOf(
            signingSDKExperiment
        ).let(::SDKExperimentsManager)
    }
    // endregion
}
