package com.instabug.library.internal.orchestrator;

import androidx.annotation.NonNull;

import com.instabug.library.Constants;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.threading.PoolProvider;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Executor;

/**
 * A helper class used to orchestrate a series of actions
 */
public final class ActionsOrchestrator {
    private final List<Action> sameThreadActions;
    private final List<Action> workerThreadActions;
    private final Executor backgroundExecutor;

    private ActionsOrchestrator(Executor backgroundExecutor) {
        this.backgroundExecutor = backgroundExecutor;
        sameThreadActions = new ArrayList<>();
        workerThreadActions = new ArrayList<>();
    }

    /**
     * Creates a new ActionsOrchestrator instance with default background executor
     */
    public static ActionsOrchestrator obtainOrchestrator() {
        return new ActionsOrchestrator(PoolProvider.getInstance().getBackgroundExecutor());
    }

    /**
     * Creates a new ActionsOrchestrator instance with the desired background executor
     */
    public static ActionsOrchestrator obtainOrchestrator(@NonNull Executor workerThreadExecutor) {
        return new ActionsOrchestrator(workerThreadExecutor);
    }

    /**
     * Add action to the current thread actions queue
     */
    public ActionsOrchestrator addSameThreadAction(@NonNull Action action) {
        sameThreadActions.add(action);
        return this;
    }

    /**
     * Add action to the background thread actions queue
     */
    public ActionsOrchestrator addWorkerThreadAction(@NonNull Action action) {
        workerThreadActions.add(action);
        return this;
    }

    /**
     * Orchestrates the execution of all pre-filled actions (mainThread and background actions).
     * This method catches and logs all exceptions
     * <p>
     * Actions will be executed in the same order they were added in as long as they were added to
     * the same actions queue (i.e. {@link #sameThreadActions}. Different queues are not guaranteed
     * to debounce in the same order
     * <p>
     * Example:
     * ActionsOrchestrator.obtainOrchestrator()
     * .addWorkerThreadAction(..) <-- Action #1 Queue #1
     * .sameThreadActions(..)     <-- Action #1 Queue #2
     * .addWorkerThreadAction(..) <-- Action #2 Queue #1
     * .sameThreadActions(..)     <-- Action #2 Queue #2
     * .orchestrate();
     * <p>
     * GUARANTEED to be executed in the same order
     * - [Action #1 Queue #1] & [Action #2 & Queue #1]
     * - [Action #1 Queue #2] & [Action #2 & Queue #2]
     * <p>
     * NOT GUARANTEED to be executed in the same order
     * - [Action #1 Queue #1] & [Action #1 & Queue #2]
     * - [Action #2 Queue #1] & [Action #2 & Queue #2]
     */
    public void orchestrate() {
        orchestrateSameThreadActions();
        orchestrateWorkerThreadActions();
    }

    private void orchestrateSameThreadActions() {
        for (Action action : sameThreadActions) {
            runAction(action);
        }
    }

    private void orchestrateWorkerThreadActions() {
        PoolProvider.postTask(backgroundExecutor, new Runnable() {
            @Override
            public void run() {
                for (Action action : workerThreadActions) {
                    runAction(action);
                }
            }
        });
    }

    private void runAction(Action action) {
        try {
            action.run();
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error while running action: " + e.getMessage(), e);
        }
    }
}
