package com.instabug.library.instacapture.screenshot.pixelcopy;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.graphics.Bitmap;
import android.os.Build;
import android.util.Pair;
import android.view.PixelCopy;
import android.view.View;
import android.view.Window;

import androidx.annotation.IdRes;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;

import com.instabug.library.Constants;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.instacapture.utility.Memory;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.threading.PoolProvider;

import java.util.HashMap;
import java.util.Map;

import io.reactivexport.Observable;
import io.reactivexport.ObservableEmitter;
import io.reactivexport.ObservableOnSubscribe;
import io.reactivexport.ObservableSource;
import io.reactivexport.android.schedulers.AndroidSchedulers;
import io.reactivexport.functions.Function;
import io.reactivexport.schedulers.Schedulers;


public class PixelCopyDelegate {

    @RequiresApi(api = Build.VERSION_CODES.O)
    public static Observable<Bitmap> takeScreenshot(@NonNull final Activity activity, @Nullable @IdRes final int[] ignoredViewsIds) {
        return Observable.create(new ObservableOnSubscribe<Bitmap>() {
                    @Override
                    public void subscribe(final ObservableEmitter<Bitmap> emitter) {
                        // create screenshot bitmap
                        final DisplaySpecs displaySpecs = new DisplaySpecs(activity);
                        try {
                            final Bitmap screenshot;
                            if ((long) displaySpecs.width * displaySpecs.height * 4 < Memory.getFreeMemory(activity)) {
                                // ARGB_8888 store each pixel in 4 bytes
                                screenshot = Bitmap.createBitmap(displaySpecs.width, displaySpecs.height, Bitmap.Config.ARGB_8888);
                            } else {
                                // RGB_565 store each pixel in 2 bytes
                                screenshot = Bitmap.createBitmap(displaySpecs.width, displaySpecs.height, Bitmap.Config.RGB_565);
                            }
                            emitter.onNext(screenshot);
                        } catch (IllegalArgumentException | OutOfMemoryError e) {
                            // log error
                            InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while capturing " + e.getMessage(), e);
                            emitter.onError(e);
                        }
                    }
                }).subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .map((Function<Bitmap, Pair<Bitmap, HashMap<View, Integer>>>) screenshot -> {
                    // hide ignored views
                    try {
                        final HashMap<View, Integer> ignoredViews = hideIgnoredViews(activity, ignoredViewsIds);
                        return new Pair<>(screenshot, ignoredViews);
                    } catch (OutOfMemoryError | Exception e) {
                        IBGDiagnostics.reportNonFatal(e, e.getMessage() != null ? "Something went wrong while hide Ignored Views " : "");
                        return new Pair<>(screenshot, new HashMap<>());
                    }
                }).observeOn(Schedulers.io())
                .flatMap(screenshotCapturingObserver(activity));
    }

    @RequiresApi(api = Build.VERSION_CODES.O)
    @NonNull
    private static Function<Pair<Bitmap, HashMap<View, Integer>>, ObservableSource<Bitmap>> screenshotCapturingObserver(@NonNull Activity activity) {
        return (Function<Pair<Bitmap, HashMap<View, Integer>>, ObservableSource<Bitmap>>) bitmapHashMapPair -> Observable
                .create(emitter -> {
                    final Bitmap screenshot = bitmapHashMapPair.first;
                    final HashMap<View, Integer> ignoredViews = bitmapHashMapPair.second;
                    try {
                        // capture screenshot
                        Window window = activity.getWindow();
                        PixelCopy.request(window, screenshot, new PixelCopy.OnPixelCopyFinishedListener() {
                            @SuppressLint("STARVATION")
                            @Override
                            public void onPixelCopyFinished(int resultCode) {
                                if (resultCode == PixelCopy.SUCCESS) {
                                    // render dialog
                                    if (SettingsManager.getInstance().getShouldCaptureDialog()) {
                                        renderVisibleDialogs(activity, resultCode, screenshot);
                                    }
//
                                } else {
                                    screenshot.recycle();
                                    String errorMessage = "Something went wrong while capturing screenshot using PixelCopy.request resultCode: " + resultCode;
                                    InstabugSDKLogger.e(Constants.LOG_TAG, errorMessage);
                                    emitter.onError(new Exception(errorMessage));
                                    reShowIgnoredViews(ignoredViews);
                                    return;
                                }
                                // reshow ignored views
                                reShowIgnoredViews(ignoredViews);
                                emitter.onNext(screenshot);
                            }
                        }, PixelCopyHandlerProvider.getHandler());

                    } catch (OutOfMemoryError | Exception e) {
                        IBGDiagnostics.reportNonFatal(e, e.getMessage() != null ? "Something went wrong while capturing " : "");
                        emitter.onError(e);
                    }
                });
    }

    private static HashMap<View, Integer> hideIgnoredViews(@NonNull Activity activity, int[] ignoredViewsIds) {


        HashMap<View, Integer> ignoredViews = new HashMap<>();
        if (activity == null) return ignoredViews;

        if (ignoredViewsIds != null) {
            for (int ignoredViewsId : ignoredViewsIds) {
                final View ignoredView = activity.findViewById(ignoredViewsId);
                if (ignoredView != null) {
                    ignoredViews.put(ignoredView, ignoredView.getVisibility());
                    ignoredView.setVisibility(View.GONE);
                }
            }
        }
        return ignoredViews;
    }

    private static void reShowIgnoredViews(final HashMap<View, Integer> ignoredViews) {
        PoolProvider.postMainThreadTask(() -> {
            if (ignoredViews == null || ignoredViews.size() == 0) return;

            for (Map.Entry<View, Integer> entry : ignoredViews.entrySet()) {
                entry.getKey().setVisibility(entry.getValue());
            }
        });
    }

    @SuppressLint("CheckResult")
    private static void renderVisibleDialogs(Activity activity, int resultCode, Bitmap screenshot) {
        DialogUiRenderer.tryRenderDialogs(activity, screenshot);

    }
}
