package com.instabug.library.firstseen;

import android.content.Context;

import androidx.annotation.NonNull;

import com.instabug.library.Constants;
import com.instabug.library.IBGNetworkWorker;
import com.instabug.library.networkv2.NetworkManager;
import com.instabug.library.networkv2.RequestResponse;
import com.instabug.library.networkv2.request.Endpoints;
import com.instabug.library.networkv2.request.Header;
import com.instabug.library.networkv2.request.Request;
import com.instabug.library.networkv2.request.RequestMethod;
import com.instabug.library.networkv2.request.RequestParameter;
import com.instabug.library.networkv2.request.RequestType;
import com.instabug.library.util.DeviceStateProvider;
import com.instabug.library.util.InstabugSDKLogger;

import org.json.JSONException;
import org.json.JSONObject;


public class FirstSeenRequestService {

    private static FirstSeenRequestService INSTANCE;
    private final NetworkManager networkManager;

    /**
     * Returns the current singleton instance of this class.
     *
     * @return singleton instance of FirstSeenRequestService
     */
    public static FirstSeenRequestService getInstance() {
        if (INSTANCE == null) {
            INSTANCE = new FirstSeenRequestService();
        }
        return INSTANCE;
    }

    private FirstSeenRequestService() {
        networkManager = new NetworkManager();
    }

    /**
     * Requesting the first_seen for the current app version.
     * it require app-version to be sent as a header
     *
     * @param context
     * @param firstSeenCallbacks
     * @throws JSONException
     */
    public void getCurrentAppVersionFirstSeen(final Context context, final Request.Callbacks<JSONObject,
            Throwable> firstSeenCallbacks) throws JSONException {
        if (context != null && firstSeenCallbacks != null) {
            InstabugSDKLogger.d(Constants.LOG_TAG, "fetch first_seen");

            // create fetch first seen request
            Request.Builder firstSeenRequestBuilder = new Request.Builder()
                    .endpoint(Endpoints.FIRST_SEEN)
                    .method(RequestMethod.GET);

            // adding required headers
            addCurrentAppVersionFirstSeenHeaders(context, firstSeenRequestBuilder);
            // build the request
            Request firstSeenRequest = firstSeenRequestBuilder.build();

            // do request with default connection timeout.
            InstabugSDKLogger.d(Constants.LOG_TAG, "First seen request started: " + firstSeenRequest);
            networkManager.doRequest(IBGNetworkWorker.CORE, RequestType.NORMAL, firstSeenRequest, new Request.Callbacks<RequestResponse, Throwable>() {

                @Override
                public void onSucceeded(RequestResponse requestResponse) {
                    if (requestResponse != null) {
                        InstabugSDKLogger.d(Constants.LOG_TAG, "getCurrentAppVersionFirstSeen Succeeded, Response code: " + requestResponse.getResponseCode());
                        InstabugSDKLogger.v(Constants.LOG_TAG, "Response: " + requestResponse);


                        if (requestResponse.getResponseCode() == RequestResponse.HttpStatusCode._2xx.OK) {
                            try {
                                if (requestResponse.getResponseBody() != null) {
                                    firstSeenCallbacks.onSucceeded(new JSONObject((String) requestResponse.getResponseBody()));
                                } else {
                                    firstSeenCallbacks.onSucceeded(new JSONObject());
                                }
                            } catch (JSONException e) {
                                InstabugSDKLogger.e(Constants.LOG_TAG, "getCurrentAppVersionFirstSeen got JSONException: " + e.getMessage(), e);
                                firstSeenCallbacks.onFailed(e);
                            }
                        } else {
                            firstSeenCallbacks.onFailed(new Throwable("getCurrentAppVersionFirstSeen got error with response code:" + requestResponse.getResponseCode()));
                        }
                    }
                }

                @Override
                public void onFailed(Throwable error) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "getCurrentAppVersionFirstSeen got error: " + error.getMessage(), error);
                    firstSeenCallbacks.onFailed(error);
                }
            });
        }
    }

    /**
     * Adding headers to currentAppVersionFirstSeen Request
     * The required headers is app-version
     *
     * @param context
     * @param firstSeenRequestBuilder
     */
    private void addCurrentAppVersionFirstSeenHeaders(@NonNull Context context, @NonNull Request.Builder firstSeenRequestBuilder) {
        // add app_version header
        firstSeenRequestBuilder.addHeader(new RequestParameter<>(Header.APP_VERSION, DeviceStateProvider.getAppVersion(context)));
    }
}
