package com.instabug.library.util;

import android.annotation.SuppressLint;
import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Canvas;
import android.graphics.Matrix;
import android.graphics.Paint;
import android.graphics.drawable.BitmapDrawable;
import android.graphics.drawable.Drawable;
import android.net.Uri;
import android.os.Handler;
import android.os.Looper;
import android.provider.MediaStore;
import android.widget.ImageView;

import androidx.annotation.DrawableRes;
import androidx.annotation.IntRange;
import androidx.annotation.Keep;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.WorkerThread;

import com.instabug.library.Constants;
import com.instabug.library.Instabug;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.internal.servicelocator.CoreServiceLocator;
import com.instabug.library.internal.storage.DiskUtils;
import com.instabug.library.internal.storage.ProcessedBytes;
import com.instabug.library.internal.storage.cache.AssetsCacheManager;
import com.instabug.library.model.AssetEntity;
import com.instabug.library.util.threading.PoolProvider;
import com.instabug.library.util.threading.ThreadUtils;

import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.FutureTask;
/**
 * The type Bitmap utils.
 *
 * @author mesbah
 */
public class BitmapUtils {
    private static final String ICON_FILE_PREFIX = "icon";

    /**
     * Gets bitmap from Uri
     *
     * @param uri the image file path
     * @return the bitmap from file path
     */
    @Nullable
    public static Bitmap getBitmapFromUri(@NonNull Uri uri) {
        Bitmap bitmap = null;
        try {
            if (Instabug.getApplicationContext() != null) {
                bitmap = MediaStore.Images.Media.getBitmap(Instabug.getApplicationContext()
                        .getContentResolver(), uri);
            }
        } catch (IOException e) {
            e.printStackTrace();
            InstabugSDKLogger.e(Constants.LOG_TAG, "getBitmapFromFilePath returns null because of" +
                    " " + e.getMessage());
        }
        return bitmap;
    }

    /**
     * Compress bitmap and save.
     * For decoding image and scales it to reduce memory consumption
     *
     * @param originalImageFile the original image file
     */
    public static void compressBitmapAndSave(@NonNull Context context, @NonNull final File originalImageFile) {
        if (context == null || originalImageFile == null) {
            return;
        }
        if (context != null) {
            try {
                compressBitmapAndSave(originalImageFile);

            } catch (Throwable e) {
                IBGDiagnostics.reportNonFatalAndLog(e, "Error occur while compress images" + e.getMessage(), Constants.LOG_TAG);
            }
        }

    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    private static void compressBitmapAndSave(File originalImageFile) {
        try {
            // Decode image size
            BitmapFactory.Options o = new BitmapFactory.Options();

            //inJustDecodeBounds=true to check dimensions only
            o.inJustDecodeBounds = true;
            FileInputStream originalImageInputStream = new FileInputStream(originalImageFile);
            BitmapFactory.decodeStream(originalImageInputStream, null, o);

            if (originalImageInputStream != null) {
                originalImageInputStream.close();
            }

            // The new size we want to scale to
            final int REQUIRED_SIZE = 900;

            // Find the correct scale value. It should be the power of 2.
            int scale = 1;
            while (o.outWidth / scale / 2 >= REQUIRED_SIZE &&
                    o.outHeight / scale / 2 >= REQUIRED_SIZE) {
                scale *= 2;
            }

            BitmapFactory.Options o2 = new BitmapFactory.Options();
            o2.inSampleSize = scale;

            FileInputStream fileInputStream = new FileInputStream(originalImageFile);
            Bitmap selectedBitmap = BitmapFactory.decodeStream(fileInputStream, null, o2);

            // here i override the original image file
            FileOutputStream outputStream = new FileOutputStream(originalImageFile);
            if (selectedBitmap != null) {
                selectedBitmap.compress(getImageMimeType(originalImageFile), 100, outputStream);
                selectedBitmap.recycle();
            }

            if (outputStream != null) {
                outputStream.close();
            }
            if (fileInputStream != null) {
                fileInputStream.close();
            }
        } catch (Exception e) {
            e.printStackTrace();
            InstabugSDKLogger.e(Constants.LOG_TAG, "bitmap doesn't " +
                    "compressed correctly " + e.getMessage());
        }
    }

    private static Bitmap.CompressFormat getImageMimeType(File file) {
        if (file.getName().contains("png")) {
            return Bitmap.CompressFormat.PNG;
        } else {
            return Bitmap.CompressFormat.JPEG;
        }
    }

    public static void loadBitmap(String localPath, ImageView imageView) {
        BitmapWorkerTask task = new BitmapWorkerTask(imageView);
        task.execute(localPath);
    }

    /**
     * @param localPath
     * @param imageView
     * @param fallbackDrawable in case any error happened while loading bitmap
     */
    public static void loadBitmapWithFallback(String localPath, ImageView imageView,
                                              @DrawableRes int fallbackDrawable) {
        BitmapWorkerTask task = new BitmapWorkerTask(imageView, fallbackDrawable);
        task.execute(localPath);
    }

    public static void loadBitmap(String localPath, ImageView imageView, float targetWidth,
                                  float targetHeight) {
        BitmapWorkerTask task = new BitmapWorkerTask(imageView, targetWidth, targetHeight);
        task.execute(localPath);
    }

    public static void loadBitmap(@NonNull String localPath, ImageView imageView,
                                  BitmapWorkerTask.OnImageLoadedListener onImageLoadedListener) {
        BitmapWorkerTask task = new BitmapWorkerTask(imageView, onImageLoadedListener);
        task.execute(localPath);
    }

    public static void loadBitmap(String localPath, ImageView imageView, float targetWidth,
                                  float targetHeight, BitmapWorkerTask.OnImageLoadedListener onImageLoadedListener) {
        BitmapWorkerTask task = new BitmapWorkerTask(imageView, targetWidth, targetHeight, onImageLoadedListener);
        task.execute(localPath);
    }

    public static Bitmap decodeSampledBitmapFromLocalPath(String imageFilePath) {

        // First decode with inJustDecodeBounds=true to check dimensions
        final BitmapFactory.Options options = new BitmapFactory.Options();
        options.inJustDecodeBounds = true;
        BitmapFactory.decodeFile(imageFilePath, options);

        // Calculate inSampleSize
        options.inSampleSize = calculateInSampleSize(options);

        // Decode bitmap with inSampleSize set
        options.inJustDecodeBounds = false;

        return BitmapFactory.decodeFile(imageFilePath, options);
    }

    public static int calculateInSampleSize(
            BitmapFactory.Options options) {
        // Raw height and width of image
        final int height = options.outHeight;
        final int width = options.outWidth;
        int inSampleSize = 1;
        int reqHeight = 500;
        int reqWidth = 500;
        if (height > reqHeight || width > reqWidth) {

            final int halfHeight = height / 2;
            final int halfWidth = width / 2;

            // Calculate the largest inSampleSize value that is a power of 2 and keeps both
            // height and width larger than the requested height and width.
            while ((halfHeight / inSampleSize) >= reqHeight
                    && (halfWidth / inSampleSize) >= reqWidth) {
                inSampleSize *= 2;
            }
        }

        return inSampleSize;
    }

    @WorkerThread
    public static Uri saveBitmapAsPNG(final Bitmap bitmap, final int quality, final File directory,
                                      final String fileNamePrefix) throws IOException {
        File screenShotFile = new File(directory, fileNamePrefix + "_"
                + System.currentTimeMillis() + ".png");
        BufferedOutputStream fos = new BufferedOutputStream(new FileOutputStream(screenShotFile));
        CompressResult result = compressQuietlyBitmap(bitmap, Bitmap.CompressFormat.PNG, quality, fos);
        fos.close();
        Uri uri = Uri.fromFile(screenShotFile);
        if (result.getLowMemory()) throw new OutOfMemoryError();
        if ((!result.getSuccess() && !result.getLowMemory())|| uri == null) throw new IOException("uri is null");
        return uri;
    }
    public static long getCompressedBitmapSize(Bitmap bitmap, int quality) {
        ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
        bitmap.compress(Bitmap.CompressFormat.PNG, quality, byteArrayOutputStream);
        return byteArrayOutputStream.size();
    }
    public static void saveBitmapAsPNG(final Bitmap bitmap, final int quality, final File directory,
                                       final String fileNamePrefix, final OnSaveBitmapCallback callback) {
        PoolProvider.postIOTask(() -> {
            try {
                Uri uri = saveBitmapAsPNG(bitmap, quality, directory, fileNamePrefix);
                callback.onSuccess(uri);
            } catch (IOException e) {
                callback.onError(e);
            }
        });
    }

    public static void saveBitmap(final Bitmap bitmap, final Context context, final
    OnSaveBitmapCallback callback) {
        PoolProvider.postIOTask(() -> {
            File attachmentDirectory = DiskUtils.getInstabugInternalDirectory(context);
            File screenShotFile = new File(attachmentDirectory, "bug_" + System.currentTimeMillis() + "_" + ".jpg");
            try {
                FileOutputStream fileOutputStream = new FileOutputStream(screenShotFile);
                BufferedOutputStream fos = new BufferedOutputStream(fileOutputStream);
                final boolean saved = compressQuietly(bitmap, Bitmap.CompressFormat.JPEG, 100, fos);
                fos.close();
                final Uri uri = Uri.fromFile(screenShotFile);

                //send callbacks on UI thread
                Handler handler = new Handler(Looper.getMainLooper());
                handler.post(new Runnable() {
                    @Override
                    public void run() {
                        if (saved && uri != null) {
                            callback.onSuccess(uri);
                        } else {
                            callback.onError(new Throwable("Uri equal null"));
                        }
                    }
                });
            } catch (IOException e) {
                callback.onError(e);
            }
        });
    }

    @Nullable
    @WorkerThread
    public static Uri saveDrawableBitmap(final Drawable drawable, final long currentTime) throws ExecutionException, InterruptedException {
        if (drawable == null) return null;

        return saveDrawableIntoFile(drawable, currentTime);
    }

    /**
     * Saves a Drawable into a file and returns the Uri of the saved file.
     *
     * @param drawable    The Drawable to be saved.
     * @param currentTime The current time used for generating a unique file name.
     * @return The Uri of the saved file, or null if the operation fails.
     */
    @Nullable
    private static Uri saveDrawableIntoFile(Drawable drawable, long currentTime) throws ExecutionException, InterruptedException {
        // Convert the Drawable to a Bitmap
        Bitmap bitmap = getBitmapFromDrawable(drawable);
        if (bitmap != null) {
            File file = getIconTargetDirectory(currentTime);
            boolean isSaved = saveBitmap(file, bitmap);
            if (isSaved) return Uri.fromFile(file);
        }
        return null;
    }

    @Nullable
    private static Bitmap getBitmapFromDrawable(Drawable drawable) throws ExecutionException, InterruptedException {
        Future<Bitmap> bitmapFuture = drawableToResizedBitmap(drawable);
        // Check for null Bitmap
        return bitmapFuture != null ? bitmapFuture.get() : null;
    }

    private static boolean saveBitmap(File file, Bitmap bitmap) {
        BufferedOutputStream fos = null;
        try {
            fos = new BufferedOutputStream(new FileOutputStream(file));
            return compressQuietly(bitmap, Bitmap.CompressFormat.PNG, 100, fos);
        } catch (IOException e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "can't compress bitmap");
        } finally {
            try {
                if (fos != null) fos.close();
            } catch (IOException e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "can't close BufferedOutputStream");
            }
        }
        return false;
    }

    @WorkerThread
    public static void loadBitmapForAsset(@Nullable Context context, String fileUrl, AssetEntity.AssetType type, final OnBitmapReady onBitmapReady) {
        if (context != null) {
            AssetEntity assetEntity = AssetsCacheManager.createEmptyEntity(context, fileUrl, type);
            AssetsCacheManager.getAssetEntity(assetEntity, new AssetsCacheManager
                    .OnDownloadFinished() {
                @Override
                public void onSuccess(final AssetEntity assetEntity) {
                    InstabugSDKLogger.d(Constants.LOG_TAG, "Asset Entity downloaded: " + assetEntity.getFile()
                            .getPath());
                    if (ThreadUtils.isCurrentThreadMain()) {
                        PoolProvider.postIOTask(() -> decodeBitmap(assetEntity, onBitmapReady));
                    } else {
                        decodeBitmap(assetEntity, onBitmapReady);
                    }
                }

                @Override
                public void onFailed(Throwable error) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Asset Entity downloading got error", error);
                    onBitmapReady.onBitmapFailedToLoad();
                }
            });
        }
    }

    @WorkerThread
    private static void decodeBitmap(AssetEntity assetEntity, OnBitmapReady onBitmapReady) {
        try (FileInputStream inputStream = new FileInputStream(assetEntity.getFile())) {
            final Bitmap bitmap = BitmapFactory.decodeStream(inputStream);
            onBitmapReady.onBitmapReady(bitmap);
        } catch (IOException e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Asset Entity downloading got " + "FileNotFoundException error", e);
            onBitmapReady.onBitmapFailedToLoad();
        }
    }

    @WorkerThread
    @NonNull
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public static File getIconTargetDirectory(long currentTime) {
        File directory = CoreServiceLocator.getReproScreenshotsCacheDir().getCurrentSpanDirectory();
        return new File(directory, ICON_FILE_PREFIX + "_" + currentTime + ".png");
    }

    /**
     * Gets bitmap from drawable object.
     *
     * @param drawable the drawable object
     * @return the bitmap from file path
     */
    @Nullable
    private static Future<Bitmap> drawableToResizedBitmap(final Drawable drawable) throws ExecutionException, InterruptedException {

        if (drawable instanceof BitmapDrawable) {
            FutureTask<Bitmap> originalBitmap = extractBitmap((BitmapDrawable) drawable);
            if (originalBitmap != null) return originalBitmap;
        }

        Context applicationContext = Instabug.getApplicationContext();
        if (applicationContext != null) {
            int maxResizableSize = DisplayUtils.dpToPxIntRounded(applicationContext.getResources(), 72);
            final int intrinsicWidth = drawable.getIntrinsicWidth();
            final int intrinsicHeight = drawable.getIntrinsicHeight();
            if (intrinsicWidth <= maxResizableSize && intrinsicHeight <= maxResizableSize) {
                return createBitMap(drawable, intrinsicWidth, intrinsicHeight);
            }
        }
        return null;
    }

    private static Future<Bitmap> createBitMap(Drawable drawable, int intrinsicWidth, int intrinsicHeight) throws ExecutionException, InterruptedException {
        final Bitmap bitmap = Bitmap.createBitmap(intrinsicWidth, intrinsicHeight, Bitmap.Config.ARGB_8888); // Single color bitmap will be created of 1x1 pixel
        final Canvas canvas = new Canvas(bitmap);
        final Drawable clonedDrawable = getClonedDrawable(drawable);
        drawDrawableIntoCanvasInMainThread(clonedDrawable, canvas);

        return PoolProvider.submitIOTask(() -> {
            float[] dimensions = getTargetDimensions(intrinsicWidth, intrinsicHeight);
            return resizeBitmap(bitmap, dimensions[0], dimensions[1]);
        });
    }

    @NonNull
    private static Drawable getClonedDrawable(Drawable drawable) {
        return drawable.getConstantState() != null ? drawable.getConstantState().newDrawable() : drawable;
    }

    @Nullable
    private static FutureTask<Bitmap> extractBitmap(BitmapDrawable drawable) {
        Bitmap originalBitmap = drawable.getBitmap();
        if (originalBitmap != null) {
            FutureTask<Bitmap> bitmapFutureTask = new FutureTask<>(() -> resizeBitmap(originalBitmap, 24, 24));
            bitmapFutureTask.run();
            return bitmapFutureTask;
        }
        return null;
    }

    private static void drawDrawableIntoCanvasInMainThread(Drawable clonedDrawable, Canvas canvas) throws ExecutionException, InterruptedException {
        PoolProvider.submitMainThreadTask(() -> {
            clonedDrawable.setBounds(0, 0, canvas.getWidth(), canvas.getHeight());
            clonedDrawable.draw(canvas);
            return canvas;
        }).get();

    }

    public static Bitmap drawableToBitmap(final Drawable drawable) {
        Bitmap outputBitmap;

        if (drawable instanceof BitmapDrawable) {
            BitmapDrawable bitmapDrawable = (BitmapDrawable) drawable;
            Bitmap originalBitmap = bitmapDrawable.getBitmap();
            if (originalBitmap != null) {
                return bitmapDrawable.getBitmap();
            }
        }

        int outputBitmapWidth = drawable.getIntrinsicWidth();
        int outputBitmapHeight = drawable.getIntrinsicHeight();

        if (outputBitmapWidth <= 0 || outputBitmapHeight <= 0) {
            //Fallback to a bitmap with size of 1x1 pixel
            outputBitmapWidth = outputBitmapHeight = 1;
        }

        outputBitmap = Bitmap.createBitmap(outputBitmapWidth, outputBitmapHeight, Bitmap.Config.ARGB_8888);

        Canvas canvas = new Canvas(outputBitmap);
        drawable.setBounds(0, 0, canvas.getWidth(), canvas.getHeight());
        drawable.draw(canvas);

        return outputBitmap;
    }

    /**
     * Scale down the large bitmap to fit in 24px X 24px keeping the original aspect ratio.
     */
    private static float[] getTargetDimensions(int realWidth, int realHeight) {
        float targetMaxDimension = 24f;
        float[] dimensions = new float[]{targetMaxDimension, targetMaxDimension};
        if (realHeight > realWidth) {
            dimensions[0] = ((float) realWidth / realHeight) * targetMaxDimension;
        } else if (realHeight < realWidth) {
            dimensions[1] = ((float) realHeight / realWidth) * targetMaxDimension;
        }
        return dimensions;
    }

    /**
     * This method scales bitmaps.
     *
     * @param bitmap       the bitmap required to be scaled.
     * @param targetWidth  the width that should be scaled to.
     * @param targetHeight the height that should be scaled to.
     * @return the scaled bitmap.
     */
    @Nullable
    public static Bitmap resizeBitmap(Bitmap bitmap, float targetWidth, float targetHeight) {
        if (bitmap == null) {
            return null;
        }
        if (targetWidth == 0 && targetHeight == 0) {
            return bitmap;
        }
        Bitmap output = Bitmap.createBitmap((int) targetWidth, (int) targetHeight, Bitmap.Config.ARGB_8888);
        if (bitmap.getWidth() < bitmap.getHeight() && targetWidth > targetHeight) {
            //portrait mode with landscape image
            return bitmap;
        } else if (bitmap.getWidth() > bitmap.getHeight() && targetWidth < targetHeight) {
            //landscape mode with portrait image
            return bitmap;
        }
        Canvas canvas = new Canvas(output);
        Matrix m = new Matrix();
        if (bitmap.getWidth() < bitmap.getHeight()) {
            //portrait picture
            m.setScale(targetWidth / bitmap.getWidth(), targetHeight / bitmap.getHeight());
        } else {
            //landscape picture
            m.setScale(targetHeight / bitmap.getHeight(), targetWidth / bitmap.getWidth());
        }
        canvas.drawBitmap(bitmap, m, new Paint());

        return output;
    }

    public static void saveBitmap(final Bitmap bitmap, final Uri imageUri, final Context context,
                                  final OnSaveBitmapCallback callback) {
        if (imageUri.getPath() != null) {
            PoolProvider.postIOTask(() -> {
                try {
                    if (imageUri.getPath() != null) {
                        final Uri uri = Uri.fromFile(new File(imageUri.getPath()));
                        OutputStream outputStream = context.getContentResolver().openOutputStream(uri);
                        if (outputStream != null) {
                            final boolean isSaved = compressQuietly(bitmap, Bitmap.CompressFormat.PNG, 100, outputStream);
                            Handler handler = new Handler(Looper.getMainLooper());
                            handler.post(new Runnable() {
                                @Override
                                public void run() {
                                    if (isSaved && callback != null) {
                                        callback.onSuccess(uri);
                                    }
                                }

                            });
                        }
                    }
                } catch (FileNotFoundException e) {
                    if (e.getMessage() != null) {
                        InstabugSDKLogger.e(Constants.LOG_TAG, "Error while saving bitmap: " + e.getMessage());
                    }
                }
            });
        }
    }

    @WorkerThread
    public static Bitmap decryptBitmap(String path) {
        ProcessedBytes processedBytes = InstabugCore.decryptOnTheFly(path);
        byte[] bitmapBytes;
        if (processedBytes.isProcessSuccessful()) {
            bitmapBytes = processedBytes.getFileBytes();
            return BitmapFactory.decodeByteArray(bitmapBytes, 0, bitmapBytes.length);
        } else {
            return Bitmap.createBitmap(1, 1, Bitmap.Config.ARGB_8888);
        }
    }

    public interface OnSaveBitmapCallback {
        void onSuccess(Uri absolutePath);

        void onError(Throwable throwable);
    }


    /**
     * Write a compressed version of the bitmap to the specified {@link java.io.OutputStream}.
     * If this returns true, the bitmap can be reconstructed by passing a
     * corresponding {@link java.io.InputStream} to BitmapFactory.decodeStream().
     *
     * @param format       The format of the compressed image
     * @param quality      Hint to the compressor, 0-100. 0 meaning compress for
     *                     small size, 100 meaning compress for max quality. Some
     *                     formats, like PNG which is lossless, will ignore the quality setting
     * @param outputStream The {@link java.io.OutputStream} to write the compressed data.
     * @return true if successfully compressed to the specified stream.
     */
    @WorkerThread
    public static boolean compressQuietly(@NonNull Bitmap bitmap, @NonNull Bitmap.CompressFormat format, @IntRange(from = 0, to = 100) int quality, @NonNull OutputStream outputStream) {
        try {
            return !bitmap.isRecycled() && bitmap.compress(format, quality, outputStream);
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error while compressing bitmap " + e.getMessage());
        }
        return false;
    }
    @WorkerThread
    public static CompressResult compressQuietlyBitmap(@NonNull Bitmap bitmap, @NonNull Bitmap.CompressFormat format, @IntRange(from = 0, to = 100) int quality, @NonNull OutputStream outputStream) {
        try {
            if (!bitmap.isRecycled() && bitmap.compress(format, quality, outputStream)){
                return new CompressResult(true,false);
            }
        }catch (OutOfMemoryError outOfMemoryError){
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error while compressing bitmap due to low memory " + outOfMemoryError.getMessage());
            return new CompressResult(false,true);
        }
        catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error while compressing bitmap " + e.getMessage());
        }
        return new CompressResult(false,false);

    }


    @Keep
    public interface OnBitmapReady {
        void onBitmapReady(@Nullable Bitmap bitmap);

        void onBitmapFailedToLoad();
    }
}

