package com.instabug.library.session;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.Size;

import com.instabug.library.ReflectionUtils;
import com.instabug.library.internal.data.IBGProperty;
import com.instabug.library.internal.data.SerializedName;
import com.instabug.library.model.session.CoreSession;
import com.instabug.library.model.session.SessionMapper;
import com.instabug.library.model.session.SessionRemoteEntity;
import com.instabug.library.model.session.SessionsBatchDTO;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

class SessionsMerger {

    @NonNull
    static SessionsBatchDTO merge(@Size(min = 2) List<CoreSession> batch) {
        Map<String, Object> commonKeys = extractCommonKeys(batch.get(0), batch.subList(1, batch.size()));
        List<SessionRemoteEntity> remoteEntities = new ArrayList<>();
        for (CoreSession session : batch) {
            Map<String, Object> uniqueSessionKeys = extractUniqueKeys(session, commonKeys.keySet());
            SessionRemoteEntity remoteEntity = SessionMapper.toRemoteEntity(session.getId(), uniqueSessionKeys);
            remoteEntities.add(remoteEntity);
        }
        return SessionMapper.toDTO(commonKeys, remoteEntities);
    }

    @NonNull
    private static Map<String, Object> extractUniqueKeys(@NonNull CoreSession session, @NonNull Set<String> commonKeys) {
        Map<String, Object> uniqueSessionKeys = new HashMap<>();
        for (Field declaredField : session.getClass().getDeclaredFields()) {
            declaredField.setAccessible(true);
            if (isInstabugField(declaredField)) {
                String serializedName = extractSerializedName(declaredField, session.isUsersPageEnabled());
                if (!commonKeys.contains(serializedName)) {
                    Object serializedValue = extractSerializedValue(declaredField, session);
                    if (serializedValue != null) {
                        uniqueSessionKeys.put(serializedName, serializedValue);
                    }
                }
            }
        }
        return uniqueSessionKeys;
    }

    @NonNull
    private static Map<String, Object> extractCommonKeys(@NonNull CoreSession sessionToTestAgainst, @NonNull List<CoreSession> sessionsUnderTest) {
        Map<String, Object> commonKeys = new HashMap<>();
        for (Field fieldToTestAgainst : sessionToTestAgainst.getClass().getDeclaredFields()) {
            fieldToTestAgainst.setAccessible(true);
            if (isInstabugField(fieldToTestAgainst)) {
                if (allContainSameField(sessionToTestAgainst, fieldToTestAgainst, sessionsUnderTest)) {
                    String serializedName = extractSerializedName(fieldToTestAgainst, sessionToTestAgainst.isUsersPageEnabled());
                    Object serializedValue = extractSerializedValue(fieldToTestAgainst, sessionToTestAgainst);
                    if (serializedValue != null) {
                        commonKeys.put(serializedName, serializedValue);
                    }
                }
            }
        }
        return commonKeys;
    }

    private static boolean isInstabugField(@NonNull Field field) {
        return field.getAnnotation(IBGProperty.class) != null;
    }

    private static boolean allContainSameField(@NonNull CoreSession instanceToTestAgainst,
                                               @NonNull Field fieldToTestAgainst,
                                               @NonNull List<CoreSession> sessionsUnderTest) {
        for (CoreSession sessionUnderTest : sessionsUnderTest) {
            Field fieldUnderTest = ReflectionUtils.getField(sessionUnderTest.getClass(), fieldToTestAgainst.getName());
            if (fieldUnderTest == null) return false;
            else {
                Object fieldToTestAgainstValue = ReflectionUtils.getValue(fieldToTestAgainst, instanceToTestAgainst);
                Object fieldUnderTestValue = ReflectionUtils.getValue(fieldUnderTest, sessionUnderTest);
                if (fieldToTestAgainstValue == null || !fieldToTestAgainstValue.equals(fieldUnderTestValue)) {
                    return false;
                }
            }
        }
        return true;
    }

    @NonNull
    private static String extractSerializedName(@NonNull Field field, boolean extractPrimaryName) {
        SerializedName annotation = field.getAnnotation(SerializedName.class);
        if (annotation == null) return field.getName();
        else if (extractPrimaryName) return annotation.name();
        else return annotation.alternate().isEmpty() ? annotation.name() : annotation.alternate();
    }

    @Nullable
    private static Object extractSerializedValue(@NonNull Field field, @NonNull CoreSession instance) {
        return ReflectionUtils.getValue(field, instance);
    }
}
