package com.instabug.library.screenshot

import android.app.Activity
import android.graphics.Bitmap
import com.instabug.library.Constants
import com.instabug.library.internal.servicelocator.CoreServiceLocator
import com.instabug.library.screenshot.instacapture.ScreenshotRequest
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.extenstions.runOrLogError
import java.util.concurrent.Callable

/**
 * A contract for any party implementing screenshot capturing capabilities.
 */
interface ScreenshotCaptor {
    /**
     * Captures a screenshot using the given [request]
     * @param request the [ScreenshotRequest] to be processed
     */
    fun capture(request: ScreenshotRequest)

    /**
     * A callback contract for screenshot capturing asynchronous result.
     */
    interface CapturingCallback {
        /**
         * A call to this indicates that a screenshot has been successfully captured.
         * @param bitmap The [Bitmap] representing the captured screenshot
         */
        fun onCapturingSuccess(bitmap: Bitmap)

        /**
         * A call to this indicates a failure while capturing screenshot.
         * @param throwable The [Throwable] error that forms the root cause of failure.
         */
        fun onCapturingFailure(throwable: Throwable)
    }
}

object ScreenshotProvider : ScreenshotCaptor {

    private var screenshotProvider: Callable<Bitmap?>? = null

    /**
     * For CP: Provides a [Callable]<[Bitmap]> to capture screenshots externally
     * @param provider when called, returns a bitmap of the screenshot taken
     */
    @JvmStatic
    fun setScreenshotProvider(provider: Callable<Bitmap?>?) {
        screenshotProvider = provider
    }

    @Synchronized
    override fun capture(request: ScreenshotRequest) {
        runOrLogError(errorMessage = "couldn't capturing screenshot") {
            val activity = request.activity.validatedActivity
            if (screenshotProvider != null && activity != null)
                captureUsingProvidedScreenCapture(activity, request.listener)
            else
                CoreServiceLocator.screenshotCapture.process(request = request)
        }.onFailure(request.listener::onCapturingFailure)
    }

    private fun captureUsingProvidedScreenCapture(
        currentActivity: Activity,
        screenshotCapturingListener: ScreenshotCaptor.CapturingCallback
    ) {
        logStartCapturingUsingProvider()
        currentActivity.runOnUiThread {
            runOrLogError(errorMessage = "error while capturing screen shot using screenshotProvider") {
                screenshotProvider
                    ?.call()
                    ?.also(screenshotCapturingListener::onCapturingSuccess)
                    ?: screenshotCapturingListener.onCapturingFailure(
                        IllegalArgumentException("Null Bitmap from Custom Screenshot Provider")
                    )
            }.onFailure(screenshotCapturingListener::onCapturingFailure)
        }
    }

    private fun logStartCapturingUsingProvider() {
        InstabugSDKLogger.i(
            Constants.LOG_TAG,
            "Using the supplied screenshotProvider to capture the screenshot"
        )
    }
}