package com.instabug.library.model.session;

import static com.instabug.library.model.session.SessionParameter.APP_TOKEN;
import static com.instabug.library.model.session.SessionParameter.APP_VERSION;
import static com.instabug.library.model.session.SessionParameter.CRASH_REPORTING_ENABLED;
import static com.instabug.library.model.session.SessionParameter.CUSTOM_ATTRIBUTES;
import static com.instabug.library.model.session.SessionParameter.CUSTOM_ATTRIBUTES_KEYS;
import static com.instabug.library.model.session.SessionParameter.DEVICE;
import static com.instabug.library.model.session.SessionParameter.DURATION;
import static com.instabug.library.model.session.SessionParameter.OS;
import static com.instabug.library.model.session.SessionParameter.SDK_VERSION;
import static com.instabug.library.model.session.SessionParameter.STARTED_AT;
import static com.instabug.library.model.session.SessionParameter.STITCHED_SESSION_LEAD;
import static com.instabug.library.model.session.SessionParameter.USER_EMAIL;
import static com.instabug.library.model.session.SessionParameter.USER_EVENTS;
import static com.instabug.library.model.session.SessionParameter.USER_EVENTS_KEYS;
import static com.instabug.library.model.session.SessionParameter.USER_NAME;
import static com.instabug.library.model.session.SessionParameter.UUID;

import android.annotation.SuppressLint;

import androidx.annotation.Keep;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.internal.data.IBGProperty;
import com.instabug.library.internal.data.SerializedName;
import com.instabug.library.model.common.ICoreSession;
import com.instabug.library.model.common.SessionVersion;

@Keep
public class CoreSession implements ICoreSession {

    @NonNull
    @IBGProperty
    @SerializedName(name = OS)
    private final String os;

    @Nullable
    @IBGProperty
    @SerializedName(name = DEVICE)
    private String device;

    @IBGProperty
    @SerializedName(name = DURATION)
    private long duration;

    @IBGProperty
    @SerializedName(name = STARTED_AT)
    private long startTimestampMicros;

    @Nullable
    @IBGProperty
    @SerializedName(name = USER_NAME)
    private String userName;

    @Nullable
    @IBGProperty
    @SerializedName(name = USER_EMAIL)
    private String userEmail;

    @Nullable
    @IBGProperty
    @SerializedName(name = SDK_VERSION)
    private String sdkVersion;

    @Nullable
    @IBGProperty
    @SerializedName(name = APP_VERSION)
    private String appVersion;

    @Nullable
    @IBGProperty
    @SerializedName(name = USER_EVENTS, alternate = USER_EVENTS_KEYS)
    private String userEvents;

    @Nullable
    @IBGProperty
    @SerializedName(name = CUSTOM_ATTRIBUTES, alternate = CUSTOM_ATTRIBUTES_KEYS)
    private String customAttributes;

    @NonNull
    @IBGProperty
    @SerializedName(name = UUID)
    private final String uuid;

    @Nullable
    @IBGProperty
    @SerializedName(name = APP_TOKEN)
    private String appToken;

    @IBGProperty
    @SerializedName(name = CRASH_REPORTING_ENABLED)
    private boolean crashReportingEnabled;

    @IBGProperty
    @SerializedName(name = STITCHED_SESSION_LEAD)
    private boolean isStitchedSessionLead;

    @SyncStatus
    private int syncStatus;
    @NonNull
    private final String id;
    private boolean usersPageEnabled;
    private long startNanoTime;
    @Nullable
    private String productionUsage;

    private CoreSession(@NonNull String id, @NonNull String uuid, @NonNull String os) {
        this.id = id;
        this.uuid = uuid;
        this.os = os;
    }

    public boolean isStitchedSessionLead() {
        return isStitchedSessionLead;
    }

    @NonNull
    @Override
    public String getUuid() {
        return uuid;
    }

    @Nullable
    public String getAppToken() {
        return appToken;
    }

    @NonNull
    @Override
    public String getId() {
        return id;
    }

    @NonNull
    @Override
    public String getOs() {
        return os;
    }

    @Nullable
    public String getDevice() {
        return device;
    }

    public long getDuration() {
        return duration;
    }

    @Override
    public long getStartTimestampMicros() {
        return startTimestampMicros;
    }

    @Override
    public long getStartNanoTime() {
        return startNanoTime;
    }

    @SessionVersion
    @Override
    public String getVersion() {
        return SessionVersion.V2;
    }

    @Nullable
    public String getUserName() {
        return userName;
    }

    @Nullable
    public String getUserEmail() {
        return userEmail;
    }

    @Nullable
    public String getSdkVersion() {
        return sdkVersion;
    }

    @Nullable
    @Override
    public String getAppVersion() {
        return appVersion;
    }

    @Nullable
    public String getUserEvents() {
        return userEvents;
    }

    @Nullable
    public String getCustomAttributes() {
        return customAttributes;
    }

    public boolean isCrashReportingEnabled() {
        return crashReportingEnabled;
    }

    public static class Builder {
        @NonNull
        private String os;
        @Nullable
        private String device;
        private long duration;
        private long startTimestampMicros;
        @Nullable
        private String userName;
        @Nullable
        private String userEmail;
        @Nullable
        private String sdkVersion;
        @Nullable
        private String appVersion;
        @Nullable
        private String userEvents;
        @Nullable
        private String customAttributes;
        @NonNull
        private String uuid;
        @Nullable
        private String appToken;
        private boolean crashReportingEnabled = false;
        private boolean isStitchedSessionLead = false;

        @SyncStatus
        private int syncStatus = SyncStatus.OFFLINE;
        @NonNull
        private String id;
        private boolean usersPageEnabled = false;
        private long startNanoTime;
        @Nullable
        private String productionUsage;

        Builder(@NonNull String id, @NonNull String uuid, @NonNull String os) {
            this.id = id;
            this.uuid = uuid;
            this.os = os;
        }

        public Builder setOs(@NonNull String os) {
            this.os = os;
            return this;
        }

        public Builder setDevice(@Nullable String device) {
            this.device = device;
            return this;
        }

        public Builder setDuration(long duration) {
            this.duration = duration;
            return this;
        }

        public Builder setStartTimestampMicros(long startTimestampMicros) {
            this.startTimestampMicros = startTimestampMicros;
            return this;
        }

        public Builder setUserName(@Nullable String userName) {
            this.userName = userName;
            return this;
        }

        public Builder setUserEmail(@Nullable String userEmail) {
            this.userEmail = userEmail;
            return this;
        }

        public Builder setSdkVersion(@Nullable String sdkVersion) {
            this.sdkVersion = sdkVersion;
            return this;
        }

        public Builder setAppVersion(@Nullable String appVersion) {
            this.appVersion = appVersion;
            return this;
        }

        public Builder setUserEvents(@Nullable String userEvents) {
            this.userEvents = userEvents;
            return this;
        }

        public Builder setCustomAttributes(@Nullable String customAttributes) {
            this.customAttributes = customAttributes;
            return this;
        }

        public Builder setUuid(@NonNull String uuid) {
            this.uuid = uuid;
            return this;
        }

        public Builder setAppToken(@Nullable String appToken) {
            this.appToken = appToken;
            return this;
        }

        public Builder setCrashReportingEnabled(boolean crashReportingEnabled) {
            this.crashReportingEnabled = crashReportingEnabled;
            return this;
        }

        public Builder setIsStitchedSessionLead(boolean isStitchedSessionLead) {
            this.isStitchedSessionLead = isStitchedSessionLead;
            return this;
        }

        public Builder setSyncStatus(@SyncStatus int syncStatus) {
            this.syncStatus = syncStatus;
            return this;
        }

        public Builder setId(@NonNull String id) {
            this.id = id;
            return this;
        }

        public Builder setUsersPageEnabled(boolean usersPageEnabled) {
            this.usersPageEnabled = usersPageEnabled;
            return this;
        }

        public Builder setStartNanoTime(long startNanoTime) {
            this.startNanoTime = startNanoTime;
            return this;
        }

        public Builder setProductionUsage(@Nullable String productionUsage) {
            this.productionUsage = productionUsage;
            return this;
        }

        @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
        public CoreSession build() {
            CoreSession coreSession = new CoreSession(id, uuid, os);
            coreSession.device = device;
            coreSession.appToken = appToken;
            coreSession.appVersion = appVersion;
            coreSession.duration = duration;
            coreSession.productionUsage = productionUsage;
            coreSession.crashReportingEnabled = crashReportingEnabled;
            coreSession.isStitchedSessionLead = isStitchedSessionLead;
            coreSession.customAttributes = customAttributes;
            coreSession.sdkVersion = sdkVersion;
            coreSession.startNanoTime = startNanoTime;
            coreSession.startTimestampMicros = startTimestampMicros;
            coreSession.syncStatus = syncStatus;
            coreSession.userEmail = userEmail;
            coreSession.userEvents = userEvents;
            coreSession.userName = userName;
            coreSession.usersPageEnabled = usersPageEnabled;
            return coreSession;
        }
    }

    @SyncStatus
    public int getSyncStatus() {
        return syncStatus;
    }

    public boolean isUsersPageEnabled() {
        return usersPageEnabled;
    }

    @Nullable
    public String getProductionUsage() {
        return productionUsage;
    }
}
