package com.instabug.library.internal.utils.stability.handler.exception;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.internal.utils.stability.execution.Executable;
import com.instabug.library.internal.utils.stability.execution.ReturnableExecutable;
import com.instabug.library.internal.utils.stability.handler.penalty.DeathPenaltyHandler;
import com.instabug.library.internal.utils.stability.handler.penalty.LogPenaltyHandler;
import com.instabug.library.internal.utils.stability.handler.penalty.PenaltyHandler;
import com.instabug.library.internal.utils.stability.handler.penalty.SwallowPenaltyHandler;

/**
 * A simple interface for handling exceptions. It acts as a try/catch block but it decouples
 * the error catching from the error handling.
 * <p>
 * Example:
 * new ExceptionHandler()
 * .withPenaltyLog()
 * .execute(() -> {
 * // unstable code
 * });
 */
public final class ExceptionHandler {

    private static final String TAG = "ExceptionHandler";
    @NonNull
    private PenaltyHandler penaltyHandler;

    public ExceptionHandler() {
        this.penaltyHandler = new SwallowPenaltyHandler();
    }

    /**
     * If the code under execution has experienced an error condition, it will be caught and logged
     * with default tag {@link #TAG}
     *
     * @return the same ExceptionHandler instance this method was called in to chain calls
     */
    @NonNull
    public ExceptionHandler withPenaltyLog() {
        this.penaltyHandler = new LogPenaltyHandler(TAG);
        return this;
    }

    /**
     * If the code under execution has experienced an error condition, it will be caught and logged
     * with the passed tag
     *
     * @param tag to be used for logging
     * @return the same ExceptionHandler instance this method was called in to chain calls
     */
    @NonNull
    public ExceptionHandler withPenaltyLog(@NonNull String tag) {
        this.penaltyHandler = new LogPenaltyHandler(tag);
        return this;
    }

    /**
     * If the code under execution has experienced an error condition, it will be caught and swallowed
     *
     * @return the same ExceptionHandler instance this method was called in to chain calls
     */
    @NonNull
    public ExceptionHandler withPenaltySwallow() {
        this.penaltyHandler = new SwallowPenaltyHandler();
        return this;
    }

    /**
     * If the code under execution has experienced an error condition. It will be caught and rethrown
     *
     * @return the same ExceptionHandler instance this method was called in to chain calls
     */
    @NonNull
    public ExceptionHandler withPenaltyDeath() {
        this.penaltyHandler = new DeathPenaltyHandler();
        return this;
    }

    /**
     * A generic helper method for attaching custom {@link PenaltyHandler}
     *
     * @return the same ExceptionHandler instance this method was called in to chain calls
     */
    @NonNull
    public ExceptionHandler withPenalty(@NonNull PenaltyHandler penaltyHandler) {
        this.penaltyHandler = penaltyHandler;
        return this;
    }

    /**
     * Executes a void block of code and delegates the error handling to the penalty handler
     *
     * @param executable the block of code to be executed
     */
    public void execute(@NonNull Executable executable) {
        try {
            executable.execute();
        } catch (Exception e) {
            penaltyHandler.handle(e);
        }
    }

    /**
     * Executes a block of code that is expected to return a value and delegates the error handling
     * to the penalty handler
     *
     * @param returnableExecutable the block of code expected to return a value
     * @return the user-specified return value if the execution was successful or null otherwise
     */
    @Nullable
    public <T> T executeAndGet(@NonNull ReturnableExecutable<T> returnableExecutable) {
        try {
            return returnableExecutable.execute();
        } catch (Exception e) {
            penaltyHandler.handle(e);
        }
        return null;
    }

    /**
     * Executes a block of code that is expected to return a value and delegates the error handling
     * to the penalty handler
     *
     * @param returnableExecutable the block of code expected to return a value
     * @param fallbackValue        the value to be returned in case the execution has experienced an error condition
     * @return the user-specified return value if the execution was successful or the passed fallback value otherwise
     */
    @NonNull
    public <T> T executeAndGet(@NonNull ReturnableExecutable<T> returnableExecutable, @NonNull T fallbackValue) {
        try {
            T output = returnableExecutable.execute();
            return output != null ? output : fallbackValue;
        } catch (Exception e) {
            penaltyHandler.handle(e);
        }
        return fallbackValue;
    }
}
