package com.instabug.library.interactionstracking

import android.content.Context
import android.graphics.Rect
import android.os.Build
import android.view.SurfaceView
import android.view.TextureView
import android.view.View
import android.view.ViewGroup
import android.webkit.WebView
import android.widget.AbsSeekBar
import android.widget.CompoundButton
import android.widget.EditText
import android.widget.GridView
import android.widget.HorizontalScrollView
import android.widget.ImageSwitcher
import android.widget.ImageView
import android.widget.ListView
import android.widget.ScrollView
import android.widget.SearchView
import android.widget.SeekBar
import android.widget.TextView
import androidx.appcompat.widget.SwitchCompat
import androidx.core.widget.NestedScrollView
import androidx.viewpager.widget.ViewPager
import com.instabug.library.model.StepType
import com.instabug.library.model.UserStep
import com.instabug.library.usersteps.UserStepsMessageGenerator
import com.instabug.library.util.InstabugDateFormatter
import com.instabug.library.util.StringUtility
import com.instabug.library.visualusersteps.TouchedView
import com.instabug.library.visualusersteps.TouchedViewsProcessor
import java.lang.ref.WeakReference
import java.util.concurrent.Future

/**
 * A concrete representation of UI nodes that uses legacy [View]s as their underlying origin.
 * Essentially wraps the origin with a [WeakReference] to guard against potential leaks.
 * All methods of the representation is throwing [IllegalArgumentException] when the underlying
 * view is nullified for any reason.
 */
abstract class BaseIBGLegacyViewUINode<CT>(
    originView: View,
    protected val nextGenTransformer: UINodeTransformer<CT>
) : IBGUINode {

    protected val originViewRef = WeakReference(originView)

    /**
     * Exposed origin view for operations that needs it.
     */
    val origin: View?
        get() = originViewRef.get()
    final override val isIBGView: Boolean
        @Throws get() = onOriginOrThrow { javaClass.name.startsWith("com.instabug") }
    final override val isVisible: Boolean
        @Throws get() = onOriginOrThrow { visibility == View.VISIBLE }
    final override val isClickable: Boolean
        @Throws get() = onOriginOrThrow { isClickable }
    final override val isLongClickable: Boolean
        @Throws get() = onOriginOrThrow { isLongClickable }
    final override val isAContainer: Boolean
        @Throws get() = onOriginOrThrow { this is ViewGroup }
    final override val zOrder: Float
        @Throws get() = onOriginOrThrow { zCompat }
    final override val isScrollable: Boolean
        @Throws get() = onOriginOrThrow { isScrollable }
    final override val isSwipeable: Boolean
        @Throws get() = onOriginOrThrow { isSwipeable }
    final override val componentClassName: String
        @Throws get() = onOriginOrThrow { javaClass.name }
    final override val isCheckable: Boolean
        @Throws get() = onOriginOrThrow { this is CompoundButton }
    final override val isChecked: Boolean
        @Throws get() = onOriginOrThrow { (this as? CompoundButton)?.isChecked ?: false }
    final override val isMovableWithProgress: Boolean
        @Throws get() = onOriginOrThrow { this is SeekBar }
    final override val isTextField: Boolean
        @Throws get() = onOriginOrThrow { this is EditText }
    final override val isFocusable: Boolean
        @Throws get() = onOriginOrThrow { isFocusable }
    final override val isPrivate: Boolean
        get() = runCatching { onOriginOrThrow { isPrivate } }.getOrDefault(false)

    @Throws
    final override fun isTouchWithinBounds(x: Float, y: Float): Boolean = onOriginOrThrow {
        val coordinates = IntArray(2)
        getLocationOnScreen(coordinates)
        val vx: Int = coordinates[0]
        val vy: Int = coordinates[1]
        val w = width
        val h = height
        !(x < vx || x > (vx + w) || y < vy || y > (vy + h))
    }

    /**
     * Adds [isPotentialOverlay] check to the original default acceptance conditions.
     */
    final override fun isValidTouchTarget(x: Float, y: Float): Boolean {
        return super.isValidTouchTarget(x, y) && !isPotentialOverlay()
    }

    @Throws
    final override fun asTouchedView(): Future<TouchedView?>? =
        onOriginOrThrow { TouchedViewsProcessor.instance.getMostProminentLabelFrom(this) }

    @Throws
    final override fun asUserStep(@StepType gesture: String, holder: Context): UserStep =
        onOriginOrThrow {
            UserStep().apply {
                val label = trimmedText
                timeStamp = InstabugDateFormatter.getCurrentUTCTimeStampInMiliSeconds()
                setType(gesture)
                message = UserStepsMessageGenerator.generateUserActionStepMessage(
                    gesture,
                    componentClassName,
                    inspectNameById(holder),
                    label,
                    holder.javaClass.name
                )
                args = UserStep.Args(type, label, componentClassName, holder.javaClass.name)
            }
        }

    @Throws
    final override fun asRect(): Rect? =
        onOriginOrThrow { Rect().takeIf { getGlobalVisibleRect(it) } }

    final override fun isIBGComponent(ibgComponentsIds: IntArray): Boolean =
        runCatching { onOriginOrThrow { ibgComponentsIds.contains(id) } }
            .getOrDefault(false)

    protected inline fun <Out> onOriginOrThrow(block: View.() -> Out): Out {
        return requireNotNull(originViewRef.get()) { "Origin View is null" }.let(block)
    }

    private val View.zCompat: Float
        get() = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) z else 0f

    private val View.isScrollable: Boolean
        get() = this is ScrollView ||
                this is HorizontalScrollView ||
                this is GridView ||
                this is ListView ||
                this is WebView ||
                this is NestedScrollView ||
                this.javaClass.name.equals("androidx.recyclerview.widget.RecyclerView") ||
                this.javaClass.name.equals("com.google.android.material.tabs.TabLayout")

    private val View.isSwipeable
        get() = this is SwitchCompat ||
                this is AbsSeekBar ||
                this is ViewPager

    private val View.trimmedText: String?
        get() = ((this as? TextView)?.text as? String)?.let { txt ->
            val trimmedText = StringUtility.trimString(txt, 15)
            return "$trimmedText${takeIf { (trimmedText.length < txt.length) }?.let { "..." } ?: ""}"
        }

    private fun View.inspectNameById(context: Context): String? = runCatching {
        id.takeIf { it != -1 }
            ?.let { requireNotNull(context.resources).getResourceEntryName(it) }
    }.getOrNull()

    /**
     * Check whether the view is a potential overlay or not
     * Currently overlays defined as:
     *  -View Groups with no children & does not have click indications
     *  -ViewGroup with one child and that one child is not a ViewGroup itself & does not have click indications
     */
    private fun isPotentialOverlay(): Boolean = runCatching {
        onOriginOrThrow {
            if (this !is ViewGroup) return false
            val isViewClickable = isClickable || isLongClickable
            if (isViewClickable) return false
            var containsGroupChild = false
            var leafChildrenCount = 0
            var leafChildIndex = -1
            for (index in 0 until childCount) {
                val child = getChildAt(index)
                if (child is ViewGroup) {
                    containsGroupChild = true
                    break
                }
                leafChildrenCount++
                leafChildIndex = index
            }
            if (containsGroupChild || leafChildrenCount > 1) return false
            if (leafChildIndex == -1) return true
            return !with(getChildAt(leafChildIndex)) { isClickable || isLongClickable }
        }
    }.getOrElse { false }
}

/**
 * A concrete implementation for non-bridging compose Views.
 */
class IBGLegacyViewUINode(
    originView: View,
    nextGenTransformer: UINodeTransformer<View>
) : BaseIBGLegacyViewUINode<View>(originView, nextGenTransformer) {
    override val childCount: Int
        @Throws get() = onOriginOrThrow { (this as? ViewGroup)?.childCount ?: 0 }
    override val type: Int
        get() = inferType()

    @Throws
    override fun getChildAt(index: Int): IBGUINode? = onOriginOrThrow {
        runCatching {
            (this as? ViewGroup)?.getChildAt(index)?.let(nextGenTransformer::transform)
        }.getOrNull()
    }

    /**
     * Infers the type of the node from the underlying origin.
     * Since [EditText] is a child of [TextView], bitwise operations is use to combine
     * the available types in case double matching.
     * For example, [EditText] should be considered an [IBGUINode.Type.LABEL] & [IBGUINode.Type.INPUT_FIELD].
     * So, the type of the node is eventually a binary combination representing both types.
     */
    private fun inferType(): Int = runCatching {
        onOriginOrThrow {
            var inferredType = IBGUINode.Type.UNDEFINED
            takeIf { it.isOfLabelType() }
                ?.also { inferredType = inferredType or IBGUINode.Type.LABEL }
            takeIf { it.isOfInputType() }
                ?.also { inferredType = inferredType or IBGUINode.Type.INPUT_FIELD }
            takeIf { it.isOfMediaType() }
                ?.also { inferredType = inferredType or IBGUINode.Type.MEDIA }
            inferredType
        }
    }.getOrDefault(IBGUINode.Type.UNDEFINED)

    private fun View.isOfLabelType(): Boolean = this is TextView

    private fun View.isOfInputType(): Boolean = this is EditText ||
            this is androidx.appcompat.widget.SearchView ||
            this is SearchView

    private fun View.isOfMediaType(): Boolean = this is ImageView ||
            this is ImageSwitcher ||
            this is SurfaceView ||
            this is TextureView
}