package com.instabug.library.diagnostics.customtraces.cache

import com.instabug.library.Constants
import com.instabug.library.diagnostics.diagnostics_db.CustomTracesAttributesEntry.COLUMN_KEY
import com.instabug.library.diagnostics.diagnostics_db.CustomTracesAttributesEntry.COLUMN_TRACE_ID
import com.instabug.library.diagnostics.diagnostics_db.CustomTracesAttributesEntry.COLUMN_VALUE
import com.instabug.library.diagnostics.diagnostics_db.CustomTracesAttributesEntry.TABLE_NAME
import com.instabug.library.diagnostics.diagnostics_db.DiagnosticsDbManager
import com.instabug.library.internal.storage.cache.dbv2.IBGContentValues
import com.instabug.library.internal.storage.cache.dbv2.IBGWhereArg
import com.instabug.library.util.InstabugSDKLogger

class AttributesDBHelperImpl : AttributesDBHelper {

    val database: DiagnosticsDbManager = DiagnosticsDbManager.getInstance()

    override fun addAttribute(traceId: Long, key: String, value: String): Boolean {
        return database.let {
            val contentValues = IBGContentValues().apply {
                put(COLUMN_TRACE_ID, traceId, ColumnsTransitiveState.TRACE_ID)
                put(COLUMN_KEY, key, ColumnsTransitiveState.KEY)
                put(COLUMN_VALUE, value, ColumnsTransitiveState.VALUE)
            }
            it.insert(TABLE_NAME, null, contentValues) != -1L
        }.also {
            InstabugSDKLogger.v(Constants.LOG_TAG, "Added custom trace attribute with $key")
        }
    }

    override fun removeAttribute(traceId: Long, key: String): Boolean {
        return database.let {
            val whereClause = "$COLUMN_TRACE_ID = ? AND $COLUMN_KEY = ?"
            val whereArgs = listOf<IBGWhereArg>(
                IBGWhereArg(traceId.toString(), ColumnsTransitiveState.TRACE_ID),
                IBGWhereArg(key, ColumnsTransitiveState.KEY)
            )
            it.delete(TABLE_NAME, whereClause, whereArgs) > 0
        }.also {
            InstabugSDKLogger.v(Constants.LOG_TAG, "Deleted custom trace attribute with $key")
        } ?: false
    }

    override fun updateAttribute(traceId: Long, key: String, value: String): Boolean {
        return database.let {
            val contentValues = IBGContentValues().apply {
                put(COLUMN_VALUE, value, ColumnsTransitiveState.VALUE)
            }
            val whereClause = "$COLUMN_TRACE_ID = ? AND $COLUMN_KEY = ?"
            val whereArgs = listOf<IBGWhereArg>(
                IBGWhereArg(traceId.toString(), ColumnsTransitiveState.TRACE_ID),
                IBGWhereArg(key, ColumnsTransitiveState.KEY)
            )
            it.update(TABLE_NAME, contentValues, whereClause, whereArgs) > 0
        }.also {
            InstabugSDKLogger.v(Constants.LOG_TAG, "Updated custom trace attribute with $key")
        } ?: false
    }

    override fun getTraceAttributes(traceId: Long): HashMap<String, String> {
        val attributes = hashMapOf<String, String>()
        database.let {
            val whereClause = "$COLUMN_TRACE_ID = ?"
            val whereArgs = listOf<IBGWhereArg>(
                IBGWhereArg(
                    traceId.toString(),
                    ColumnsTransitiveState.TRACE_ID
                )
            )

            database.query(TABLE_NAME, null, whereClause, whereArgs, null, null, null)?.use { cursor ->
                while (cursor.moveToNext()) {
                    attributes[cursor.getString(cursor.getColumnIndex(COLUMN_KEY))] = cursor.getString(
                        cursor.getColumnIndex(
                            COLUMN_VALUE
                        )
                    )
                }
            }
        }

        return attributes
    }
}