package com.instabug.library.datahub

import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent
import com.instabug.library.internal.filestore.CurrentSpanSelector
import com.instabug.library.internal.filestore.DeleteDirectory
import com.instabug.library.internal.filestore.MakeDirectoryWithAncestors
import com.instabug.library.internal.filestore.OldSpansSelector
import com.instabug.library.internal.filestore.OldestSpanOnLimitSelector
import com.instabug.library.internal.filestore.OperationScopeBuilder
import com.instabug.library.internal.filestore.SpansDirectoryFactory
import com.instabug.library.util.extenstions.deleteRecursivelyDefensive
import com.instabug.library.util.extenstions.getOrLogAndReport
import com.instabug.library.util.extenstions.logDWithThreadName
import com.instabug.library.util.extenstions.logVWithThreadName
import com.instabug.library.util.extenstions.runOrLogAndReport
import com.instabug.library.util.threading.OrderedExecutorService
import java.util.concurrent.Future

interface HubController {
    fun init(launchId: String)
    fun onNewEvent(event: IBGSdkCoreEvent)
    fun cleanse()
    fun shutdown(): Future<Boolean>
}

private const val MAX_NUMBER_OF_LAUNCHES = 100

class DefaultHubController(
    private val executor: OrderedExecutorService,
    private val directoryFactory: SpansDirectoryFactory<HubDirectory>,
    private val storesList: List<HubControlledDataStore<*>>
) : HubController {

    private var hubDirectory: HubDirectory? = null

    override fun init(launchId: String) {
        executor.execute(HubExecutionQueues.Controller) {
            runCatching {
                "[Hub] Hub is being initialized with ALID $launchId".logDWithThreadName()
                directoryFactory.setCurrentSpanId(launchId)
                hubDirectory = directoryFactory()?.apply {
                    "[Hub] ALID directory is being created $this".logVWithThreadName()
                    OperationScopeBuilder(MakeDirectoryWithAncestors<HubLaunchDirectory>())
                        .onSpan(CurrentSpanSelector())
                        .buildAndExecute(this)
                    "[Hub] Directories is being trimmed to $MAX_NUMBER_OF_LAUNCHES limit".logVWithThreadName()
                    OperationScopeBuilder(DeleteDirectory<HubLaunchDirectory>())
                        .onSpan(OldestSpanOnLimitSelector(MAX_NUMBER_OF_LAUNCHES))
                        .buildAndExecute(this)
                    "[Hub] Registered stores is being initialized with $this".logVWithThreadName()
                    storesList.map { store -> store.init(this) }
                        .forEach(Future<Boolean>::get)
                } ?: run { "[Hub] Directory creation produced null.".logDWithThreadName(); null }
            }.runOrLogAndReport("[Hub] Error while initializing hub controller.")
        }
    }

    override fun onNewEvent(event: IBGSdkCoreEvent) {
        executor.execute(HubExecutionQueues.Controller) {
            runCatching {
                "[Hub] Controller received new event $event".logDWithThreadName()
                "[Hub] Delegating event to registered stores ...".logVWithThreadName()
                storesList.filterIsInstance<HubEventsListener>()
                    .forEach { listener -> listener.onNewEvent(event) }
            }.runOrLogAndReport("[Hub] Error while delegating new event to hub data stores.")
        }
    }

    override fun cleanse() {
        executor.execute(HubExecutionQueues.Controller) {
            runCatching {
                "[Hub] Controller is being cleansed.".logDWithThreadName()
                "[Hub] Delegating cleansing command to registered store ...".logVWithThreadName()
                storesList.map(HubControlledDataStore<*>::cleanse)
                    .forEach(Future<Boolean>::get)
                "[Hub] Deleting old spans directories ...".logVWithThreadName()
                OperationScopeBuilder(DeleteDirectory<HubLaunchDirectory>())
                    .onMultiSpans(OldSpansSelector())
                    .buildAndExecute(hubDirectory)
            }.runOrLogAndReport("[Hub] Error while cleansing hub data stores.")
        }
    }

    override fun shutdown(): Future<Boolean> =
        executor.submit(HubExecutionQueues.Controller) {
            runCatching {
                "[Hub] Controller is being shutdown.".logDWithThreadName()
                "[Hub] Delegating shutdown command to registered stores ...".logVWithThreadName()
                storesList.map(HubControlledDataStore<*>::shutdown)
                    .forEach(Future<Boolean>::get)
                "[Hub] Deleting entire hub directory ...".logVWithThreadName()
                hubDirectory?.deleteRecursivelyDefensive()
                hubDirectory = null; true
            }.getOrLogAndReport(false, "[Hub] Error while shutting down data hub.")
        }
}