package com.instabug.library.util.threading

import java.util.concurrent.ThreadFactory

/**
 * @author mhashim6 on 31/3/22
 *
 * Instantiates [Thread]s with a standard naming scheme: "IBG-(identifier)-(thread num)".
 *
 * Parentheses are not included.
 */
open class IBGThreadFactory constructor(
    val identifier: String?
) :
    ThreadFactory {
    private var threadNum = 0
    fun increment() = threadNum++

    private inline fun createThread(
        runnable: Runnable,
        crossinline onInterrupt: Thread.() -> Unit = {}
    ) =
        object : Thread(runnable) {
            override fun run() {
                try {
                    rename(identifier)
                    super.run()
                } catch (_: Throwable ) {
                    // Do nothing
                }
            }

            override fun interrupt() {
                try {
                    super.interrupt()
                    onInterrupt()
                } catch (_: Throwable ) {
                        // Do nothing
                    }
            }
        }


    override fun newThread(runnable: Runnable): Thread? = createThread(runnable)

    fun Thread.rename(identifier: String?) = apply {
        name = generateThreadName(identifier, threadNum)
        increment()
    }
}

/**
 * @return a [String] following the naming scheme: "IBG-(identifier)-(thread num)".
 */
fun generateThreadName(identifier: String?, number: Int) = "IBG-$identifier-$number"
