package com.instabug.library.util.filters;


import static com.instabug.library.Constants.LOG_TAG;

import android.annotation.SuppressLint;
import android.util.Pair;

import androidx.annotation.NonNull;

import com.instabug.library.internal.servicelocator.CoreServiceLocator;
import com.instabug.library.internal.storage.cache.db.userAttribute.UserAttributeCacheManager;
import com.instabug.library.model.UserAttribute;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.filters.actions.Action;
import com.instabug.library.util.filters.actions.ReturnableAction;

import java.util.HashMap;
import java.util.Map;

public final class AttributeFiltersFunctions {


    public static Filter<String> nonBackEndKeyFilter() {
        return new Filter<String>() {
            @Override
            @SuppressLint("ERADICATE_RETURN_NOT_NULLABLE")
            public String apply(String key) {
                int type = UserAttributeCacheManager.getType(key);
                return type == UserAttribute.Type.BACKEND ? null : key;
            }
        };
    }

    public static Filter<HashMap<String, String>> nonBackEndMapFilter() {
        return new Filter<HashMap<String, String>>() {
            @Override
            public HashMap<String, String> apply(HashMap<String, String> map) {
                HashMap<String, String> filteredStringMap = new HashMap<>();
                if (map != null) {
                    for (Map.Entry<String, String> entry : map.entrySet()) {
                        if (UserAttributeCacheManager.getType(entry.getKey()) != UserAttribute.Type.BACKEND
                                && isValidAttribute(entry.getKey(), entry.getValue()))
                            filteredStringMap.put(entry.getKey(), entry.getValue());
                    }
                }
                return filteredStringMap;
            }
        };
    }

    public static boolean isValidAttribute(UserAttribute attribute) {
        return isValidAttribute(attribute.getKey(), attribute.getValue());
    }

    public static boolean isValidAttribute(@NonNull String key, @NonNull String value) {
        int userAttributesCharactersLimit = CoreServiceLocator.getUserAttributesCharactersLimit();
        boolean isValid = (key.length() <= userAttributesCharactersLimit) &&
                (value.length() <= userAttributesCharactersLimit);
        if (!isValid)
            InstabugSDKLogger.w(LOG_TAG, "Some user attributes weren't added. Max allowed user attributes characters limit is reached. " +
                    "Please note that you can add user attributes (key, value) with characters count up to " + userAttributesCharactersLimit + " characters.");
        return isValid;
    }

    public static Filter<Pair<String, String>> nonBackEndKeyValuePairFilter() {
        return new Filter<Pair<String, String>>() {
            @Override
            @SuppressLint("ERADICATE_RETURN_NOT_NULLABLE")
            public Pair<String, String> apply(Pair<String, String> keyValuePair) {
                int type = UserAttributeCacheManager.getType(keyValuePair.first);
                return type == UserAttribute.Type.BACKEND ? null : keyValuePair;
            }
        };
    }

    public static Action<String> deleteAttributeAction() {
        return new Action<String>() {
            @Override
            public void apply(String key) {
                if (key != null) {
                    UserAttributeCacheManager.delete(key);
                }
            }
        };
    }

    public static Action<Pair<String, String>> insertAttributeAction() {
        return new Action<Pair<String, String>>() {
            @Override
            public void apply(Pair<String, String> keyValuePair) {
                if (keyValuePair != null) {
                    if (keyValuePair.second == null) {
                        InstabugSDKLogger.e(LOG_TAG, "Passed null value to UserAttribute key: " + keyValuePair.first + ".Ignoring this attribute.");
                        return;
                    }

                    if (isValidAttribute(keyValuePair.first, keyValuePair.second))
                        UserAttributeCacheManager.insert(keyValuePair.first, keyValuePair.second);
                }
            }
        };
    }

    public static Action<HashMap<String, String>> deleteAttributesAction() {
        return new Action<HashMap<String, String>>() {
            @Override
            public void apply(HashMap<String, String> map) {
                if (map != null) {
                    for (Map.Entry<String, String> entry : map.entrySet()) {
                        UserAttributeCacheManager.delete(entry.getKey());
                    }
                }
            }
        };
    }

    public static ReturnableAction<String, String> attributeValueReturnableAction() {
        return new ReturnableAction<String, String>() {
            @Override
            @SuppressLint("ERADICATE_RETURN_NOT_NULLABLE")
            public String apply(String key) {
                if (key == null) return null;
                String value = UserAttributeCacheManager.retrieve(key);
                if (value == null) return null;
                return isValidAttribute(key, value) ? value : null;

            }
        };
    }
}
