package com.instabug.library.tracking;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.library.core.InstabugCore;
import com.instabug.library.logscollection.LogDistributor;
import com.instabug.library.model.StepType;
import com.instabug.library.model.UserStep;
import com.instabug.library.sessionreplay.di.SessionReplayServiceLocator;
import com.instabug.library.usersteps.UserStepsMessageGenerator;
import com.instabug.library.util.InstabugDateFormatter;
import com.instabug.library.util.threading.PoolProvider;

import java.util.concurrent.Future;

import kotlin.Unit;

/**
 * @author mesbah.
 */
public class InstabugTrackingStepsProvider {


    public final static int USER_STEPS_LIMIT = 100;
    @Nullable
    private static InstabugTrackingStepsProvider INSTANCE;
    private final LogDistributor<UserStep> distributor;
    private static final String USER_STEPS_KEY = "user_steps";

    private InstabugTrackingStepsProvider() {
        distributor = LogDistributor.createBasicLogDistributorFor(
                SessionReplayServiceLocator.getCoreUserStepReceiver()
        );
    }

    /**
     * Returns the current singleton instance of this class.
     *
     * @return singleton instance of InstabugTrackingStepsProvider.
     */
    public synchronized static InstabugTrackingStepsProvider getInstance() {
        if (INSTANCE == null) {
            INSTANCE = new InstabugTrackingStepsProvider();
        }
        return INSTANCE;
    }

    /**
     * Add tracking step.
     *
     * @param holderName step holder name.
     * @param stepType   step type
     * @see StepType
     */
    public Future<Unit> addActivityLifecycleStep(final String holderName, @StepType final String stepType) {
        return PoolProvider.submitOrderedIOTask(USER_STEPS_KEY, () -> {
            if (InstabugCore.isForegroundBusy())
                return Unit.INSTANCE;

            UserStep step = createUserStep(stepType);
            step.setMessage(UserStepsMessageGenerator
                    .generateActivityLifecycleStepMessage(stepType, holderName));
            fillStepArgs(step, holderName);
            distributeUserStep(step);
            return Unit.INSTANCE;
        });
    }

    private void distributeUserStep(UserStep step) {
        distributor.invoke(step);
    }

    /**
     * Add tracking step.
     *
     * @param holderName       step holder name.
     * @param holderParentName step holder parent name.
     * @param stepType         step type
     * @see StepType
     */
    public void addFragmentLifecycleStep(String holderName, String holderParentName, @StepType String stepType) {
        addFragmentLifecycleStep(holderName, holderParentName, null, stepType);
    }

    /**
     * Add tracking step.
     *
     * @param holderName       step holder name.
     * @param holderParentName step holder parent name.
     * @param extraInfo        step extra info.
     * @param stepType         step type
     */
    public void addFragmentLifecycleStep(final String holderName, final String holderParentName,
                                         @Nullable final String extraInfo, @StepType final String stepType) {
        PoolProvider.postOrderedIOTask(USER_STEPS_KEY, new Runnable() {
            @Override
            public void run() {
                if (InstabugCore.isForegroundBusy())
                    return;
                UserStep step = createUserStep(stepType);
                step.setMessage(UserStepsMessageGenerator
                        .generateFragmentLifecycleStepMessage(stepType, holderName,
                                holderParentName, extraInfo));
                fillStepArgs(step, holderName);

                distributeUserStep(step);
            }
        });

    }

    private static void fillStepArgs(UserStep step, @Nullable String holderName) {
        fillStepArgs(step, holderName, null, null);
    }

    private static void fillStepArgs(UserStep step, @Nullable String holderName, @Nullable String uiClass, @Nullable String view) {
        step.setArgs(new UserStep.Args(step.getType(), holderName, uiClass, view));
    }

    private UserStep createUserStep(@StepType String stepType) {
        UserStep userStep = new UserStep();
        userStep.setTimeStamp(InstabugDateFormatter.getCurrentUTCTimeStampInMiliSeconds());
        userStep.setType(stepType);
        return userStep;
    }

    /**
     * Add tapping step.
     *
     * @param message
     */
    public void addViewTouchStep(@StepType String stepType, String message,
                                 String uiClass, @Nullable String label, String view) {
        UserStep step = createUserStep(stepType);
        step.setMessage(message);
        fillStepArgs(step, label, uiClass, view);
        distributeUserStep(step);

    }

    public void addUserStep(@NonNull UserStep step){
        distributeUserStep(step);
    }

    @VisibleForTesting
    static synchronized void reset() {
        INSTANCE = null;
    }
}
